package cn.zhxu.toys.msg;


import cn.zhxu.toys.util.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * 可感知当前平台的消息发送器
 *
 * @author Troy.Zhou
 *
 * @since 0.5.8
 *
 */
public class PlatformMsgSender implements MsgSender {

    /**
     * 消息发送器池，一个平台一个
     */
    private final Map<Object, AbstractMsgSender> senderPool = new HashMap<>();

    private final Object lock = new Object();

    private Map<String, String[]> templates;

    private PlatformSensor platformSensor;

    private SenderFactory senderFactory;

    /**
     * 平台消息发送器钩子
     */
    public interface PlatformSensor {

        /**
         * 感知当前平台的主键标识
         * @return 平台主键标识
         */
        Object sensePlatformKey();

        /**
         * 感知当前平台的消息发送器的初始化参数
         * @return 初始化参数
         */
        Map<String, String> senseInitParams();

        /**
         * 感知当前平台的模板 ID
         * @param tmplName 模板名称
         * @return 模板值
         */
        String senseTmplValue(String tmplName);

    }

    /**
     * 消息发送器工厂
     */
    public interface SenderFactory {

        /**
         * 创建一个消息发送器
         * @return 消息发送器
         */
        AbstractMsgSender create();

    }

    @Override
    public boolean send(String phone, String tmplName, String... tmplArgs) {
        Object key = platformSensor.sensePlatformKey();
        AbstractMsgSender sender;
        MsgTemplate template;
        synchronized (lock) {
            sender = senderPool.get(key);
            if (sender == null) {
                sender = senderFactory.create();
                if (sender == null) {
                    throw new IllegalStateException("the return of MsgSenderFactory#create() can not be null !");
                }
                sender.init(platformSensor.senseInitParams());
                senderPool.put(key, sender);
            }
            template = sender.getMsgTemplate(tmplName);
            if (template == null) {
                String tmplValue = platformSensor.senseTmplValue(tmplName);
                if (StringUtils.isBlank(tmplValue)) {
                    throw new IllegalStateException("the return of PlatformSensor#senseMsgTmplValue(" + tmplName + ") is NULL!");
                }
                String[] argsNames = templates.get(tmplName);
                if (argsNames == null) {
                    throw new IllegalStateException("No such template named: " + tmplName);
                }
                template = new MsgTemplate(tmplName, tmplValue, argsNames);
                sender.addMsgTemplate(template);
            }
        }
        return sender.send(phone, template, tmplArgs);
    }

    public Map<String, String[]> getTemplates() {
        return templates;
    }

    public void setTemplates(Map<String, String[]> templates) {
        this.templates = templates;
    }

    public PlatformSensor getPlatformSensor() {
        return platformSensor;
    }

    public void setPlatformSensor(PlatformSensor platformSensor) {
        this.platformSensor = platformSensor;
    }

    public SenderFactory getSenderFactory() {
        return senderFactory;
    }

    public void setSenderFactory(SenderFactory senderFactory) {
        this.senderFactory = senderFactory;
    }

}
