package cn.zhxu.toys.push;

import java.util.Map;

import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import cn.jiguang.common.ClientConfig;
import cn.jpush.api.JPushClient;
import cn.jpush.api.push.model.Message;
import cn.jpush.api.push.model.Message.Builder;
import cn.jpush.api.push.model.Platform;
import cn.jpush.api.push.model.PushPayload;
import cn.jpush.api.push.model.audience.Audience;
import cn.jpush.api.push.model.notification.Notification;

/**
 * 
 * 极光推送管理器
 * 
 * @author Troy.Zhou
 *
 */
public class JPushManager implements PushManager, InitializingBean, DisposableBean {

	static Logger log = LoggerFactory.getLogger(JPushManager.class);
	
	
	private String appKey;
	private String masterSecret;
	private boolean testing;
	
	
	private JPushClient jpushClient = null;
	

	@Override
	public void afterPropertiesSet() throws Exception {
		ClientConfig config = ClientConfig.getInstance();
		config.setApnsProduction(!testing);
		jpushClient = new JPushClient(masterSecret, appKey, null, config);
	}

	
	@Override
	public PushResult pushToAll(String title, String message) {
		return pushToAll(title, message, null);
	}

	@Override
	public PushResult pushToAll(String title, String message, Map<String, Object> data) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.all())
				.setAudience(Audience.all())
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}

	
	@Override
	public PushResult pushToUser(String title, String message, String... userId) {
		return pushToUser(title, message, null, userId);
	}

	@Override
	public PushResult pushToUser(String title, String message, Map<String, Object> data, String... userId) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.all())
				.setAudience(Audience.alias(userId))
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}
	
	@Override
	public PushResult pushToAndroid(String title, String message) {
		return pushToAndroid(title, message, null);
	}

	
	@Override
	public PushResult pushToAndroid(String title, String message, Map<String, Object> data) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.android())
				.setAudience(Audience.all())
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}

	
	@Override
	public PushResult pushToAndroidUser(String title, String message, String... userId) {
		return pushToAndroidUser(title, message, null, userId);
	}


	@Override
	public PushResult pushToAndroidUser(String title, String message, Map<String, Object> data, String... userId) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.android())
				.setAudience(Audience.alias(userId))
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}

	
	@Override
	public PushResult pushToIos(String title, String message) {
		return pushToIos(title, message, null);
	}


	@Override
	public PushResult pushToIos(String title, String message, Map<String, Object> data) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.ios())
				.setAudience(Audience.all())
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}


	@Override
	public PushResult pushToIosUser(String title, String message, String... userId) {
		return pushToIosUser(title, message, null, userId);
	}


	@Override
	public PushResult pushToIosUser(String title, String message, Map<String, Object> data, String... userId) {
		Builder builder = Message.newBuilder().setTitle(title).setMsgContent(message);
		if (data != null) {
			data.forEach((String key, Object value) -> {
				builder.addExtra(key, String.valueOf(value));
			});
		}
		cn.jpush.api.push.model.PushPayload.Builder pBuilder = PushPayload.newBuilder()
				.setPlatform(Platform.ios())
				.setAudience(Audience.alias(userId))
				.setMessage(builder.build());
		if (title != null) {
			pBuilder.setNotification(Notification.alert(title));
		}
		return pushPayload(pBuilder.build());
	}
	
	
	@Override
	public void destroy() {
		if (jpushClient != null) {
			jpushClient.close();
		}
	}

	private PushResult pushPayload(PushPayload payload) {
		if (jpushClient == null) {
			throw new PushException("Jpush Client 没有实例化！");
		}
		try {
			cn.jpush.api.push.PushResult result = jpushClient.sendPush(payload);
			String message = result.getOriginalContent();
			if (result.statusCode == HttpStatus.SC_OK) {
				return PushResult.success("200", message);
			}
			log.error("极光推送失败：" + message);
			return PushResult.fail(String.valueOf(result.statusCode), message);
		} catch (Exception e) {
			throw new PushException("极光推送异常：", e);
		}
	}

	
	public void setAppKey(String appKey) {
		this.appKey = appKey;
	}

	public void setMasterSecret(String masterSecret) {
		this.masterSecret = masterSecret;
	}

	public void setTesting(boolean testing) {
		this.testing = testing;
	}

}

