package cn.ziyicloud.framework.boot.autoconfigure.data.mongo.listener;

import cn.ziyicloud.framework.boot.autoconfigure.data.mongo.annotation.ZiyiAutoIncrement;
import cn.ziyicloud.framework.boot.autoconfigure.data.mongo.entity.Sequence;
import org.springframework.data.mongodb.core.FindAndModifyOptions;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.mapping.event.AbstractMongoEventListener;
import org.springframework.data.mongodb.core.mapping.event.BeforeConvertEvent;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.util.ReflectionUtils;

/**
 * 保存文档监听类<br>
 * 在保存对象时，通过反射方式为其生成ID
 *
 * @author Li Ruitong
 * @date 2020/5/25
 */
public class SaveEventListener extends AbstractMongoEventListener<Object> {
    private final MongoTemplate mongoTemplate;

    public SaveEventListener(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    @Override
    public void onBeforeConvert(BeforeConvertEvent<Object> event) {
        Object source = event.getSource();
        ReflectionUtils.doWithFields(source.getClass(), field -> {
            ReflectionUtils.makeAccessible(field);
            // 如果字段添加了我们自定义的AutoIncKey注解
            if (field.isAnnotationPresent(ZiyiAutoIncrement.class)) {
                Object id = field.get(source);
                if (id == null) {
                    // 设置自增ID
                    field.set(source, String.valueOf(getNextId(source.getClass().getSimpleName())));
                }
            }
        });

        super.onBeforeConvert(event);
    }

    /**
     * 获取下一个自增ID
     *
     * @param collName 集合（这里用类名，就唯一性来说最好还是存放长类名）名称
     * @return 序列值
     */
    private Long getNextId(String collName) {
        Query query = new Query(Criteria.where("collName").is(collName));
        Update update = new Update();
        update.inc("incId", 1);
        FindAndModifyOptions options = new FindAndModifyOptions();
        options.upsert(true);
        options.returnNew(true);
        Sequence seq = mongoTemplate.findAndModify(query, update, options, Sequence.class);
        return seq.getIncId();
    }
}
