package cn.ziyicloud.framework.boot.autoconfigure.web.handler;

import cn.ziyicloud.framework.boot.enums.ErrorCodeEnum;
import cn.ziyicloud.framework.boot.exception.ZiyiException;
import cn.ziyicloud.framework.boot.model.ZiyiResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import java.util.List;

/**
 * 统一异常 处理
 *
 * @author Li Ruitong
 * @date 2020/5/26
 */
@Slf4j
@RestControllerAdvice
@Order(1)
public class ZiyiExceptionHandler {
    /**
     * 拦截自定义异常进行处理
     *
     * @param e ZiyiBootException
     * @return ZiyiBootResponse
     */
    @ExceptionHandler(ZiyiException.class)
    public ZiyiResponse exceptionHandler(ZiyiException e) {
        debugLog(e);
        return new ZiyiResponse(e);
    }

    /**
     * post参数解析错误
     */
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ZiyiResponse exceptionHandler(HttpMessageNotReadableException e) {
        debugLog(e);
        return new ZiyiResponse(new ZiyiException(ErrorCodeEnum.BAD_REQUEST, e.getMessage(), e));
    }

    /**
     * 请求方法不支持
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public ZiyiResponse exceptionHandler(HttpRequestMethodNotSupportedException e) {
        debugLog(e);
        return new ZiyiResponse(new ZiyiException(ErrorCodeEnum.METHOD_NOT_SUPPORTED, e));
    }

    /**
     * 请求Content type不支持
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public ZiyiResponse exceptionHandler(HttpMediaTypeNotSupportedException e) {
        debugLog(e);
        return new ZiyiResponse(new ZiyiException(ErrorCodeEnum.MEDIA_TYPE_NOT_SUPPORTED, e));
    }

    /**
     * 参数校验错误
     */
    @ExceptionHandler(BindException.class)
    public ZiyiResponse exceptionHandler(BindException e) {
        debugLog(e);
        List<ObjectError> objectErrors = e.getBindingResult().getAllErrors();
        return getValidExceptionResult(objectErrors);
    }

    /**
     * 参数校验错误
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ZiyiResponse exceptionHandler(MethodArgumentNotValidException e) {
        debugLog(e);
        List<ObjectError> objectErrors = e.getBindingResult().getAllErrors();
        return getValidExceptionResult(objectErrors);
    }


    /**
     * 拦截Exception异常
     */
    @ExceptionHandler(Exception.class)
    public ZiyiResponse exceptionHandler(Exception e) {
        debugLog(e);
        return new ZiyiResponse(new ZiyiException(ErrorCodeEnum.BAD_REQUEST, e.getMessage(), e));
    }

    /**
     * 获取参数校验异常响应代码
     */
    private ZiyiResponse getValidExceptionResult(List<ObjectError> objectErrors) {
        StringBuilder sb = new StringBuilder();
        for (ObjectError error : objectErrors) {
            sb.append(error.getDefaultMessage()).append(";");
        }
        String message = sb.length() > 0 ? sb.toString().substring(0, sb.length() - 1) : sb.toString();
        return new ZiyiResponse(ErrorCodeEnum.PARAMETER_NOT_SUPPORTED, message);
    }

    /**
     * 记录日志
     */
    private void debugLog(Exception e) {
        if (log.isDebugEnabled()) {
            log.debug("响应异常", e);
        }
    }
}
