package cn.ziyicloud.framework.boot.autoconfigure.web.handler;

import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.RSA;
import cn.ziyicloud.framework.boot.autoconfigure.web.ZiyiWebProperties;
import cn.ziyicloud.framework.boot.autoconfigure.web.annotation.ZiyiRSAEncrypt;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

/**
 * 统一响应 拦截加密
 *
 * @author Li Ruitong
 * @date 2020/5/26
 */
@Slf4j
@RestControllerAdvice
@Order(2)
public class ZiyiRSAEncryptHandler implements ResponseBodyAdvice<Object> {
    private ZiyiWebProperties.RsaInfo rsaInfo;
    private RSA rsa;
    private ObjectMapper objectMapper = new ObjectMapper();


    public ZiyiRSAEncryptHandler(ZiyiWebProperties ziyiWebProperties) {
        this.rsaInfo = ziyiWebProperties.getRsa();
        this.rsa = new RSA(ziyiWebProperties.getRsa().getPrivateKey(), ziyiWebProperties.getRsa().getPublicKey());
    }

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        if (!rsaInfo.isOpen()) {
            return false;
        }
        //含有ZiyiBootRSAEncrypt注解的类或方法
        return (AnnotatedElementUtils.hasAnnotation(returnType.getContainingClass(), ZiyiRSAEncrypt.class) || returnType.hasMethodAnnotation(ZiyiRSAEncrypt.class));
    }

    @Override
    public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType, Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest request, ServerHttpResponse response) {
        String privateKey = rsaInfo.getPrivateKey();
        try {
            String content = objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(body);
            //私钥加密
            return rsa.encryptBcd(content, KeyType.PublicKey);
        } catch (Exception e) {
            log.error("加密数据异常", e);
        }
        return body;
    }
}
