package cn.ziyicloud.framework.boot.autoconfigure.knife4j;

import com.github.xiaoymin.knife4j.spring.filter.ProductionSecurityFilter;
import com.github.xiaoymin.knife4j.spring.filter.SecurityBasicAuthFilter;
import com.github.xiaoymin.knife4j.spring.model.MarkdownFiles;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.servlet.config.annotation.InterceptorRegistration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import springfox.bean.validators.configuration.BeanValidatorPluginsConfiguration;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.configuration.SwaggerCommonConfiguration;
import springfox.documentation.swagger2.configuration.Swagger2DocumentationConfiguration;
import springfox.documentation.swagger2.web.Swagger2ControllerWebMvc;

import java.lang.reflect.Field;
import java.util.List;

import static cn.ziyicloud.framework.boot.autoconfigure.knife4j.ZiyiKnife4jProperties.ZIYI_BOOT_API_PREFIX;

/**
 * Swagger2+Knife4j自动化配置
 *
 * @author Li Ruitong
 * @date 2020/5/26
 */
@Slf4j
@Configuration
@EnableConfigurationProperties({ZiyiKnife4jProperties.class})
@ConditionalOnClass({SwaggerCommonConfiguration.class, Swagger2DocumentationConfiguration.class, Swagger2ControllerWebMvc.class})
@ConditionalOnProperty(prefix = ZIYI_BOOT_API_PREFIX, name = "enable", havingValue = "true", matchIfMissing = true)
@ComponentScan(basePackages = {"com.github.xiaoymin.knife4j.spring.plugin"})
@Import({Swagger2DocumentationConfiguration.class, BeanValidatorPluginsConfiguration.class})
public class ZiyiKnife4jAutoConfiguration implements WebMvcConfigurer {
    /**
     * swagger+knife4j相关属性配置
     */
    private final ZiyiKnife4jProperties knife4jProperties;
    private final ZiyiKnife4jProperties.ZiyiSwaggerProperties swaggerProperties;
    /**
     * spring bean factory
     */
    private final BeanFactory beanFactory;

    public ZiyiKnife4jAutoConfiguration(ZiyiKnife4jProperties knife4jProperties, BeanFactory beanFactory) {
        this.knife4jProperties = knife4jProperties;
        this.swaggerProperties = knife4jProperties.getSwagger();
        this.beanFactory = beanFactory;
    }

    /**
     * 配置swagger基本信息
     * - BasePackage
     * 默认使用SpringBoot项目扫描bean根目录
     * 如果存在配置时则使用SwaggerProperties.basePackage作为扫描根目录
     *
     * @return Docket实例
     */
    @Bean
    public Docket docket() {
        String basePackage = swaggerProperties.getBasePackage();
        if (StringUtils.isEmpty(basePackage)) {
            basePackage = AutoConfigurationPackages.get(beanFactory).get(0);
        }
        return new Docket(DocumentationType.SWAGGER_2)
            .apiInfo(apiInfo())
            .select()
            .apis(RequestHandlerSelectors.basePackage(basePackage))
            .paths(PathSelectors.any())
            .build();
    }

    /**
     * 初始化自定义Markdown特性
     *
     * @return markdownFiles
     */
    @Bean(initMethod = "init")
    public MarkdownFiles markdownFiles() {
        return new MarkdownFiles(knife4jProperties.getMarkdowns() == null ? "" : knife4jProperties.getMarkdowns());
    }

    @Bean
    public SecurityBasicAuthFilter securityBasicAuthFilter() {
        return knife4jProperties.getBasic().toKnife4jAuthFilter();
    }

    @Bean
    public ProductionSecurityFilter productionSecurityFilter() {
        return new ProductionSecurityFilter(knife4jProperties.isProduction());
    }

    /**
     * 配置文档基本信息
     * 如：文档标题、版本、描述、联系人基本信息等
     *
     * @return ApiInfo实例
     */
    private ApiInfo apiInfo() {
        return new ApiInfoBuilder()
            .title(swaggerProperties.getTitle())
            .description(swaggerProperties.getDescription())
            .version(swaggerProperties.getVersion())
            .license(swaggerProperties.getLicense())
            .licenseUrl(swaggerProperties.getLicenseUrl())
            .contact(swaggerProperties.getContact().toSwaggerContact())
            .build();
    }

    /**
     * 通用拦截器排除swagger设置，所有拦截器都会自动加swagger相关的资源排除信息
     */
    @SuppressWarnings("unchecked")
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        try {
            Field registrationsField = FieldUtils.getField(InterceptorRegistry.class, "registrations", true);
            List<InterceptorRegistration> registrations = (List<InterceptorRegistration>) ReflectionUtils.getField(registrationsField, registry);
            if (registrations != null) {
                for (InterceptorRegistration interceptorRegistration : registrations) {
                    interceptorRegistration
                        .excludePathPatterns("/swagger**/**")
                        .excludePathPatterns("/webjars/**")
                        .excludePathPatterns("/v2/**")
                        .excludePathPatterns("/v3/**")
                        .excludePathPatterns("/META-INF/resources/webjars/**")
                        .excludePathPatterns("/doc.html");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
