package cn.ziyicloud.framework.boot.autoconfigure.fastjson.filter;

import cn.ziyicloud.framework.boot.autoconfigure.fastjson.filter.annotation.ZiyiValueHide;
import cn.ziyicloud.framework.boot.util.reflect.ReflectUtils;
import com.alibaba.fastjson.serializer.ValueFilter;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;

/**
 * 数值隐藏过滤器
 *
 * @author Li Ruitong
 * @date 2020/5/15
 */
@Slf4j
public class ValueHideFilter implements ValueFilter {
    @Override
    public Object process(Object object, String name, Object value) {
        if (value != null) {
            try {
                Field field = ReflectUtils.getField(object, name);
                //带ZiyiValueHide注解的字段
                if (field.isAnnotationPresent(ZiyiValueHide.class)) {
                    ZiyiValueHide valueHide = field.getDeclaredAnnotation(ZiyiValueHide.class);
                    // don't execute hide
                    if (valueHide.length() <= 0) {
                        return object;
                    }
                    // field value
                    String fieldValue = String.valueOf(value);
                    // convert to char array
                    char[] chars = fieldValue.toCharArray();
                    // hide with start position
                    if (valueHide.start() > 0) {
                        return hide(valueHide, chars);
                    }
                    // hide with position wrapper
                    else {
                        return positionHide(valueHide, chars);
                    }
                }
            } catch (Exception e) {
                log.debug("数值隐藏出错", e);
            }
        }
        return value;
    }

    /**
     * 隐藏字符
     *
     * @param valueHide ZiyiValueHide
     * @param chars     chars
     * @return 隐藏后的字符串
     */
    private String hide(ZiyiValueHide valueHide, char[] chars) {
        for (int i = 0; i < chars.length; i++) {
            if (valueHide.start() - 1 <= i && i < valueHide.start() + valueHide.length() - 1) {
                chars[i] = valueHide.placeholder();
            }
        }
        return new String(chars);
    }

    /**
     * 位置隐藏--当start <=0时
     *
     * @param valueHide ZiyiValueHide
     * @param chars     chars
     * @return 隐藏后的字符串
     */
    private String positionHide(ZiyiValueHide valueHide, char[] chars) {
        int startPosition = chars.length / 2;
        boolean hide;
        int hideLength = valueHide.length();
        char placeholder = valueHide.placeholder();
        for (int i = 0; i < chars.length; i++) {
            //默认从中间开始隐藏
            hide = i >= startPosition - 1 && i < startPosition - 1 + hideLength;
            switch (valueHide.position()) {
                case START:
                    hide = i < hideLength;
                    break;
                case END:
                    hide = i >= chars.length - hideLength;
                    break;
                default:
                    break;
            }
            if (hide) {
                chars[i] = placeholder;
            }
        }
        return new String(chars);
    }

}
