package cn.ziyicloud.framework.boot.autoconfigure.data.mongo.listener;

import cn.ziyicloud.framework.boot.autoconfigure.data.mongo.annotation.AutoIncrement;
import cn.ziyicloud.framework.boot.autoconfigure.data.mongo.entity.Sequence;
import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedDate;
import org.springframework.data.mongodb.core.FindAndModifyOptions;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.mapping.event.AbstractMongoEventListener;
import org.springframework.data.mongodb.core.mapping.event.BeforeConvertEvent;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.util.ReflectionUtils;

import java.time.LocalDateTime;

/**
 * 保存文档监听类<br>
 * 在保存对象时，通过反射方式为其生成ID
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.0
 */
public class SaveEventListener extends AbstractMongoEventListener<Object> {

    public static final String ID_TYPE_STRING = "class java.lang.String";
    public static final String ID_TYPE_LONG = "class java.lang.Long";
    public static final String ID_TYPE_INTEGER = "class java.lang.Integer";

    private final MongoTemplate mongoTemplate;

    public SaveEventListener(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    @Override
    public void onBeforeConvert(BeforeConvertEvent<Object> event) {
        Object source = event.getSource();
        ReflectionUtils.doWithFields(source.getClass(), field -> {
            ReflectionUtils.makeAccessible(field);
            // 如果字段添加了我们自定义的AutoIncKey注解
            if (field.isAnnotationPresent(AutoIncrement.class)) {
                Object idValue = field.get(source);
                // id字段值为null
                if (idValue == null) {
                    // id字段类型
                    String type = field.getGenericType().toString();
                    // id值
                    Long nextId = getNextId(source.getClass().getSimpleName());
                    switch (type) {
                        case ID_TYPE_STRING:
                            field.set(source, String.valueOf(nextId));
                            break;
                        case ID_TYPE_INTEGER:
                            field.set(source, nextId.intValue());
                            break;
                        case ID_TYPE_LONG:
                            field.set(source, nextId);
                            break;
                        default:
                    }
                }
            }
            // 审计--时间和日期
            if (field.isAnnotationPresent(CreatedDate.class)) {
                Object value = field.get(source);
                if (value == null) {
                    field.set(source, LocalDateTime.now());
                }
            }
            if (field.isAnnotationPresent(LastModifiedDate.class)) {
                field.set(source, LocalDateTime.now());
            }
        });

        super.onBeforeConvert(event);
    }

    /**
     * 获取下一个自增ID
     *
     * @param collName 集合（这里用类名，就唯一性来说最好还是存放长类名）名称
     * @return 序列值
     */
    private Long getNextId(String collName) {
        Query query = new Query(Criteria.where("collName").is(collName));
        Update update = new Update();
        update.inc("incId", 1);
        FindAndModifyOptions options = new FindAndModifyOptions();
        options.upsert(true);
        options.returnNew(true);
        Sequence seq = mongoTemplate.findAndModify(query, update, options, Sequence.class);
        assert seq != null;
        return seq.getIncId();
    }
}
