package cn.ziyicloud.framework.boot.autoconfigure.data.mongo.repository;

import cn.ziyicloud.framework.boot.autoconfigure.data.core.domain.SelfRelevance;
import cn.ziyicloud.framework.boot.enums.ErrorCodeEnum;
import cn.ziyicloud.framework.boot.exception.BaseException;
import cn.ziyicloud.framework.boot.util.ObjectUtils;
import cn.ziyicloud.framework.boot.util.conlection.CollectionUtils;

import java.util.Collection;
import java.util.List;

/**
 * 自关联表 扩展mongo repository
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.7
 */
public interface SelfRelevanceMongoRepository<T extends SelfRelevance<ID>, ID> extends BaseMongoRepository<T, ID> {
    /**
     * 根据 ids 查询
     *
     * @param ids ids
     * @return {@link List}<{@link T}> 结果列表
     */
    List<T> findByIdIn(Iterable<ID> ids);

    /**
     * 根据 pid 查询是否存在
     *
     * @param pid pid
     * @return {@literal true} pid存在, {@literal false} otherwise.
     */
    boolean existsByPid(ID pid);

    /**
     * 根据 PID 查询
     *
     * @param pid pid
     * @return {@link List}<{@link T}> 结果列表
     */
    List<T> findByPid(ID pid);

    /**
     * 根据 pids 查询
     *
     * @param pids pids
     * @return {@link List}<{@link T}> 结果列表
     */
    List<T> findByPidIn(Iterable<ID> pids);

    /**
     * 校验pid
     *
     * @param pid    上级节点id
     * @param id     节点id
     * @param rootId 根节点id
     */
    default void checkPid(ID pid, ID id, ID rootId) {
        boolean exists = null == pid || rootId == pid || existsById(pid);
        if (!exists) {
            throw new BaseException(ErrorCodeEnum.PID_NOT_EXIST);
        } else {
            if (id != null && id.equals(pid)) {
                throw new BaseException(ErrorCodeEnum.PID_IS_SELF);
            }
        }
    }

    /**
     * 获取所有子节点
     *
     * @param ids          ids 要获取子节点的id列表
     * @param childrenList 子节点列表
     * @return {@link List}<{@link T}> 子节点列表
     */
    default List<T> getAllSub(Collection<ID> ids, List<T> childrenList) {
        // 根据pid查询所有子节点
        List<T> children = findByPidIn(ids);
        // 递归结束条件
        if (children.isEmpty()) {
            return childrenList;
        }
        // 查询到的子节点id
        ids = ObjectUtils.getIds(children);
        // 查询到的子节点ids与上级子节点ids作差
        ids = CollectionUtils.subtractions(ids, ObjectUtils.getIds(childrenList));
        // 叠加
        childrenList.addAll(children);
        // 去重
        childrenList = ObjectUtils.distinctById(childrenList);
        //  递归
        return getAllSub(ids, childrenList);
    }

    /**
     * 获取所有上级节点
     *
     * @param pids       上级节点id
     * @param parentList 上级节点列表
     * @return {@link List}<{@link T}> 上级节点列表
     */
    default List<T> getAllParent(Collection<ID> pids, List<T> parentList) {
        List<T> parent = findByIdIn(pids);
        if (parent.isEmpty()) {
            return parentList;
        }
        // 查询到的上级节点pids
        pids = ObjectUtils.getPids(parent);
        // 查询到的上级节点pids与上级节点pids作差
        pids = CollectionUtils.subtractions(pids, ObjectUtils.getPids(parentList));
        // 叠加
        parentList.addAll(parent);
        // 去重
        parentList = ObjectUtils.distinctById(parentList);
        //  递归
        return getAllParent(pids, parentList);
    }
}
