package cn.ziyicloud.framework.boot.autoconfigure.data.redis;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.ParserConfig;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.jsontype.impl.LaissezFaireSubTypeValidator;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cache.CacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.*;

import java.nio.charset.StandardCharsets;
import java.time.Duration;

/**
 * Configuration  for redis.
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.7
 */
@Slf4j
@Getter
@Setter
@Configuration
@EnableConfigurationProperties(ZiyiRedisProperties.class)
@ConditionalOnClass({RedisOperations.class})
@AutoConfigureBefore(RedisAutoConfiguration.class)
public class ZiyiRedisAutoConfiguration {
    @Value("${spring.redis.database:0}")
    private int defaultDbIndex;

    private RedisTemplate<String, Object> template;

    private final ZiyiRedisProperties properties;

    public ZiyiRedisAutoConfiguration(ZiyiRedisProperties properties) {
        this.properties = properties;
    }

    @Bean
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        //key序列化用string
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashKeySerializer(new StringRedisSerializer());
        //值序列化
        RedisSerializer<Object> valueSerializer = getValueSerializer();
        redisTemplate.setValueSerializer(valueSerializer);
        redisTemplate.setHashValueSerializer(valueSerializer);

        redisTemplate.setConnectionFactory(redisConnectionFactory);
        this.setTemplate(redisTemplate);
        RedisUtils.init(template, defaultDbIndex);
        return redisTemplate;
    }

    /**
     * 缓存管理器
     *
     * @param lettuceConnectionFactory 连接工厂
     * @return {@link CacheManager} 缓存管理器
     */
    @Bean
    @ConditionalOnMissingBean
    public CacheManager cacheManager(RedisConnectionFactory lettuceConnectionFactory) {
        RedisCacheConfiguration defaultCacheConfig = RedisCacheConfiguration.defaultCacheConfig();
        // 设置缓存管理器管理的缓存的默认过期时间
        defaultCacheConfig = defaultCacheConfig.entryTtl(Duration.ofSeconds(properties.getDefaultExpireTime()))
            // 设置 key为string序列化
            .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(new StringRedisSerializer()))
            // 设置value为json序列化
            .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(getValueSerializer()))
            // 不缓存空值
            .disableCachingNullValues();
        // Set<String> cacheNames = new HashSet<>();
        // cacheNames.add(testCacheName);
        // // 对每个缓存空间应用不同的配置
        // Map<String, RedisCacheConfiguration> configMap = new HashMap<>();
        // configMap.put(testCacheName, defaultCacheConfig.entryTtl(Duration.ofSeconds(testExpireTime)));
        return RedisCacheManager.builder(lettuceConnectionFactory)
            .cacheDefaults(defaultCacheConfig)
            // .initialCacheNames(cacheNames)
            // .withInitialCacheConfigurations(configMap)
            .build();
    }

    /**
     * 获取值序列化器
     *
     * @return {@link RedisSerializer}<{@link Object}>
     */
    private RedisSerializer<Object> getValueSerializer() {
        if (properties.isFastJson()) {
            log.info("用FastJson做为redis值的序列化");
            // 全局开启AutoType，这里方便开发，使用全局的方式
            ParserConfig.getGlobalInstance().setAutoTypeSupport(true);
            return new FastJsonRedisSerializer<>(Object.class);
        }
        ObjectMapper mapper = new ObjectMapper();
        mapper.activateDefaultTyping(LaissezFaireSubTypeValidator.instance, ObjectMapper.DefaultTyping.NON_FINAL,
            JsonTypeInfo.As.PROPERTY);
        // mapper.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.registerModule(new JavaTimeModule());
        return new GenericJackson2JsonRedisSerializer(mapper);
    }

    /**
     * FastJson序列化器
     *
     * @author Li Ruitong 86415270@qq.com
     */
    static class FastJsonRedisSerializer<T> implements RedisSerializer<T> {
        private final byte[] EMPTY_ARRAY = new byte[0];

        private final Class<T> clazz;

        public FastJsonRedisSerializer(Class<T> clazz) {
            super();
            this.clazz = clazz;
        }

        @Override
        public byte[] serialize(T t) throws SerializationException {
            if (t == null) {
                return EMPTY_ARRAY;
            }
            return JSON.toJSONString(t, SerializerFeature.WriteClassName).getBytes(StandardCharsets.UTF_8);
        }

        @Override
        public T deserialize(byte[] bytes) throws SerializationException {
            if (bytes == null || bytes.length <= 0) {
                return null;
            }
            String str = new String(bytes, StandardCharsets.UTF_8);
            return JSON.parseObject(str, clazz);
        }
    }
}
