package cn.ziyicloud.framework.boot.autoconfigure.doc.springdoc;

import cn.ziyicloud.framework.boot.autoconfigure.doc.core.*;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.info.License;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springdoc.webmvc.core.SpringDocWebMvcConfiguration;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.annotation.Nonnull;

import static cn.ziyicloud.framework.boot.autoconfigure.doc.core.Constants.SWAGGER_AUTO_CONFIG;

/**
 * springdoc+Knife4j-springdoc-ui自动化配置
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.7
 */
@Slf4j
@Configuration
@EnableConfigurationProperties({ZiyiSpringdocProperties.class, ZiyiSwaggerProperties.class})
@ConditionalOnClass({SpringDocWebMvcConfiguration.class})
@RequiredArgsConstructor
public class ZiyiSpringdocAutoConfiguration implements WebMvcConfigurer {
    /**
     * spring bean factory
     */
    private final BeanFactory beanFactory;

    /**
     * Ziyi Boot 自定义OpenAPI
     *
     * @param swaggerProperties swagger 配置信息
     * @return {@link OpenAPI}
     */
    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = SWAGGER_AUTO_CONFIG, havingValue = "true", matchIfMissing = true)
    public OpenAPI ziyiOpenApi(ZiyiSwaggerProperties swaggerProperties) {
        return new OpenAPI()
            .info(getInfo(swaggerProperties.getApiInfo()));
    }

    /**
     * 通用拦截器排除swagger设置，所有拦截器都会自动加swagger相关的资源排除信息
     */
    @Override
    public void addInterceptors(@Nonnull InterceptorRegistry registry) {
        InterceptorUtils.excludeSwagger(registry);
    }

    /**
     * 获取信息
     *
     * @param apiInfo api信息
     * @return {@link Info}
     */
    private Info getInfo(ZiyiApiInfo apiInfo) {
        return new Info()
            .title(apiInfo.getTitle())
            .description(apiInfo.getDescription())
            .version(apiInfo.getVersion())
            .contact(getContact(apiInfo.getContact()))
            .license(getLicense(apiInfo.getLicense()))
            .extensions(apiInfo.getExtensions())
            ;
    }

    /**
     * 获取联系人
     *
     * @param contact 联系
     * @return {@link Contact}
     */
    private Contact getContact(ZiyiContact contact) {
        return new Contact()
            .name(contact.getName())
            .url(contact.getUrl())
            .email(contact.getEmail())
            .extensions(contact.getExtensions())
            ;
    }

    /**
     * 获取许可证
     *
     * @param license 许可证
     * @return {@link License}
     */
    private License getLicense(ZiyiLicense license) {
        return new License()
            ;
    }
}
