package cn.ziyicloud.framework.boot.autoconfigure.doc.springfox;

import cn.ziyicloud.framework.boot.autoconfigure.doc.core.InterceptorUtils;
import cn.ziyicloud.framework.boot.autoconfigure.doc.core.ZiyiApiInfo;
import cn.ziyicloud.framework.boot.autoconfigure.doc.core.ZiyiSwaggerProperties;
import com.github.xiaoymin.knife4j.spring.extension.OpenApiExtensionResolver;
import com.github.xiaoymin.knife4j.spring.filter.ProductionSecurityFilter;
import com.github.xiaoymin.knife4j.spring.filter.SecurityBasicAuthFilter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;
import org.springframework.web.filter.CorsFilter;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Contact;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.configuration.Swagger2DocumentationWebMvcConfiguration;

import javax.annotation.Nonnull;

import static cn.ziyicloud.framework.boot.autoconfigure.doc.core.Constants.*;

/**
 * Swagger2+Knife4j自动化配置
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.0
 * @since 1.0.0
 */
@Slf4j
@Configuration
@EnableConfigurationProperties({ZiyiSpringfoxProperties.class, ZiyiSwaggerProperties.class})
@ConditionalOnClass({Swagger2DocumentationWebMvcConfiguration.class})
@Import({Swagger2DocumentationWebMvcConfiguration.class})
@ComponentScan(basePackages = {KNIFE4J_SPRING_PLUGIN_PACKAGE})
@RequiredArgsConstructor
public class ZiyiSpringfoxAutoConfiguration implements WebMvcConfigurer {
    /**
     * spring bean factory
     */
    private final BeanFactory beanFactory;


    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = SWAGGER_AUTO_CONFIG, havingValue = "true", matchIfMissing = true)
    public Docket ziyiApi(ZiyiSpringfoxProperties knife4jProperties, ZiyiSwaggerProperties swaggerProperties) {
        String basePackage = swaggerProperties.getBasePackage();
        if (StringUtils.isBlank(basePackage)) {
            basePackage = AutoConfigurationPackages.get(beanFactory).get(0);
        }
        return new Docket(DocumentationType.SWAGGER_2)
            .host(swaggerProperties.getHost())
            .groupName(swaggerProperties.getGroupName())
            .apiInfo(getApiInfo(swaggerProperties.getApiInfo()))
            .select()
            .apis(RequestHandlerSelectors.basePackage(basePackage))
            .paths(PathSelectors.any())
            .build()
            .extensions(knife4jProperties.getExtensionResolver().buildExtensions(swaggerProperties.getGroupName()));
    }

    @Bean("ziyiKnife4jCorsFilter")
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = KNIFE4J_SPRINGFOX_CORS, havingValue = "true")
    public CorsFilter corsFilter() {
        log.info("init ziyiKnife4jCorsFilter...");
        UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
        CorsConfiguration corsConfiguration = new CorsConfiguration();
        corsConfiguration.setAllowCredentials(true);
        corsConfiguration.addAllowedOrigin("*");
        corsConfiguration.addAllowedHeader("*");
        corsConfiguration.addAllowedMethod("*");
        corsConfiguration.setMaxAge(10000L);
        //匹配所有API
        source.registerCorsConfiguration("/**", corsConfiguration);
        return new CorsFilter(source);
    }

    @Bean(initMethod = "start", value = "ziyiKnife4jOpenApiExtensionResolver")
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = KNIFE4J_SPRINGFOX_PLUS_ENABLED, havingValue = "true")
    public OpenApiExtensionResolver markdownResolver(ZiyiSpringfoxProperties knife4jProperties) {
        return knife4jProperties.getExtensionResolver();
    }

    @Bean("ziyiKnife4jSecurityBasicAuthFilter")
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = KNIFE4J_SPRINGFOX_BASIC_ENABLED, havingValue = "true")
    public SecurityBasicAuthFilter securityBasicAuthFilter(ZiyiSpringfoxProperties knife4jProperties) {
        return knife4jProperties.getBasic().getAuthFilter();
    }


    @Bean("ziyiKnife4jProductionSecurityFilter")
    @ConditionalOnMissingBean
    @ConditionalOnProperty(name = KNIFE4J_SPRINGFOX_PRODUCTION, havingValue = "true")
    public ProductionSecurityFilter productionSecurityFilter(ZiyiSpringfoxProperties knife4jProperties) {
        return new ProductionSecurityFilter(knife4jProperties.isProduction());
    }


    /**
     * 通用拦截器排除swagger设置，所有拦截器都会自动加swagger相关的资源排除信息
     *
     * @param registry 注册表
     */
    @Override
    public void addInterceptors(@Nonnull InterceptorRegistry registry) {
        InterceptorUtils.excludeSwagger(registry);
    }

    private ApiInfo getApiInfo(ZiyiApiInfo apiInfo) {
        return new ApiInfoBuilder()
            .title(apiInfo.getTitle())
            .description(apiInfo.getDescription())
            .version(apiInfo.getVersion())
            .termsOfServiceUrl(apiInfo.getTermsOfServiceUrl())
            .contact(new Contact(apiInfo.getContact().getName(), apiInfo.getContact().getUrl(), apiInfo.getContact().getEmail()))
            .license(apiInfo.getLicense().getName())
            .licenseUrl(apiInfo.getLicense().getUrl())
            .build();
    }

}
