package cn.ziyicloud.framework.boot.autoconfigure.fastjson;

import cn.ziyicloud.framework.boot.util.reflect.ReflectUtils;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.alibaba.fastjson.serializer.ValueFilter;
import com.alibaba.fastjson.support.config.FastJsonConfig;
import com.alibaba.fastjson.support.spring.FastJsonHttpMessageConverter;
import lombok.extern.slf4j.Slf4j;
import org.reflections.util.ConfigurationBuilder;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.autoconfigure.http.HttpMessageConvertersAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.MediaType;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.util.ObjectUtils;

import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 配置fastjson作为数据返回转换
 * 如果没有配置spring.http.converters.preferred-json-mapper参数则使用该配置进行转换数据返回
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.0
 */
@Slf4j
@Configuration
@ConditionalOnClass({FastJsonHttpMessageConverter.class, ConfigurationBuilder.class})
@EnableConfigurationProperties(ZiyiFastJsonProperties.class)
@AutoConfigureBefore(HttpMessageConvertersAutoConfiguration.class)
@ConditionalOnProperty(name = "ziyi.fast-json.http-message-converter.enable", havingValue = "true")
public class ZiyiFastJsonHttpMessageConvertersAutoConfiguration {
    private final ZiyiFastJsonProperties properties;
    /**
     * bean工厂
     */
    private final BeanFactory beanFactory;

    /**
     * value filter package
     */
    private static final String[] VALUE_FILTER_PACKAGE = new String[]{"cn.ziyicloud.framework.boot.autoconfigure.fastjson.filter"};

    public ZiyiFastJsonHttpMessageConvertersAutoConfiguration(ZiyiFastJsonProperties properties, BeanFactory beanFactory) {
        this.properties = properties;
        this.beanFactory = beanFactory;
        log.info("FastJsonHttpMessageConvertersAutoConfiguration");
    }

    @Bean
    @ConditionalOnMissingBean
    public HttpMessageConverters fastJsonHttpMessageConverters() {
        // fastJson 转换器
        FastJsonHttpMessageConverter fastConverter = new FastJsonHttpMessageConverter();
        //创建fastJson配置实体类
        FastJsonConfig fastJsonConfig = new FastJsonConfig();

        //驼峰转下划线
        if (properties.isSnakeCase()) {
            //驼峰转下划线
            SerializeConfig serializeConfig = new SerializeConfig();
            serializeConfig.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
            fastJsonConfig.setSerializeConfig(serializeConfig);
        }

        //序列化格式
        fastJsonConfig.setSerializerFeatures(getSerializerFeatures());

        // 处理中文乱码问题
        List<MediaType> fastMediaTypes = new ArrayList<>();
        // fastMediaTypes.add(MediaType.APPLICATION_JSON_UTF8);
        fastMediaTypes.add(new MediaType("application", "json", StandardCharsets.UTF_8));
        fastConverter.setSupportedMediaTypes(fastMediaTypes);

        //设置过滤器
        // 获取自定义过滤器
        List<String> packages = AutoConfigurationPackages.get(beanFactory);
        // 获取插件定义的过滤器
        packages.addAll(Arrays.asList(VALUE_FILTER_PACKAGE));
        fastJsonConfig.setSerializeFilters(getDefineFilters(packages));

        //转换器装载配置
        fastConverter.setFastJsonConfig(fastJsonConfig);

        //处理字符串, 避免直接返回字符串的时候被添加了引号
        StringHttpMessageConverter stringConverter = new StringHttpMessageConverter(StandardCharsets.UTF_8);

        return new HttpMessageConverters(fastConverter, stringConverter);
    }


    /**
     * 获取序列化格式
     *
     * @return SerializerFeature数组
     */
    private SerializerFeature[] getSerializerFeatures() {
        List<SerializerFeature> serializerFeatures = new ArrayList<>();
        //禁用 对象循环引用
        serializerFeatures.add(SerializerFeature.DisableCircularReferenceDetect);
        //格式化输出
        serializerFeatures.add(SerializerFeature.PrettyFormat);
        if (properties.isSerializerMap()) {
            serializerFeatures.add(SerializerFeature.WriteMapNullValue);
        }
        if (properties.isSerializerEnum()) {
            serializerFeatures.add(SerializerFeature.WriteEnumUsingName);
        }
        if (properties.isSerializerList()) {
            serializerFeatures.add(SerializerFeature.WriteNullListAsEmpty);
        }
        if (properties.isSerializerString()) {
            serializerFeatures.add(SerializerFeature.WriteNullStringAsEmpty);
        }
        if (properties.isSerializerNumber()) {
            serializerFeatures.add(SerializerFeature.WriteNullNumberAsZero);
        }
        if (properties.isSerializerBoolean()) {
            serializerFeatures.add(SerializerFeature.WriteNullBooleanAsFalse);
        }
        if (properties.isSerializerSkip()) {
            serializerFeatures.add(SerializerFeature.SkipTransientField);
        }
        if (properties.isSerializerSort()) {
            serializerFeatures.add(SerializerFeature.SortField);
        }
        if (properties.isSerializerClass()) {
            serializerFeatures.add(SerializerFeature.WriteClassName);
        }
        if (properties.isSerializerDate()) {
            serializerFeatures.add(SerializerFeature.WriteDateUseDateFormat);
        }
        if (properties.isSerializerSlash()) {
            serializerFeatures.add(SerializerFeature.WriteSlashAsSpecial);
        }
        if (properties.isSerializerDefault()) {
            serializerFeatures.add(SerializerFeature.NotWriteDefaultValue);
        }
        return serializerFeatures.toArray(new SerializerFeature[]{});
    }

    /**
     * 获取项目中定义的ValueFilter实现类列表
     * 通过BeanFactory读取本项目的Base Package List
     *
     * @return ValueFilter数组
     */
    @SuppressWarnings("unchecked")
    private ValueFilter[] getDefineFilters(List<String> packages) {
        Set<Class> filterClass = new HashSet<>();
        if (ObjectUtils.isEmpty(packages)) {
            return new ValueFilter[]{};
        }
        // 读取所有package下的ValueFilter实现类
        packages.forEach(pack -> filterClass.addAll((Collection<? extends Class>) ReflectUtils.getSubClassList(pack,
            ValueFilter.class)));
        List<ValueFilter> filters = new LinkedList<>();
        filterClass.forEach(filter -> {
            try {
                filters.add((ValueFilter) filter.newInstance());
            } catch (Exception e) {
                log.error("ValueFilter new instance have error.", e);
            }
        });
        log.info("Loaded ValueFilter : {}", filterClass.toString());
        return filters.toArray(new ValueFilter[]{});
    }
}
