package cn.ziyicloud.framework.boot.autoconfigure.fastjson.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.parser.ParserConfig;
import com.alibaba.fastjson.serializer.SerializeConfig;

/**
 * json转换工具类
 *
 * @author Li Ruitong 86415270@qq.com
 */
public class FastJsonUtils {
    private FastJsonUtils() {
    }

    /**
     * javabean输出字符串key值为驼峰式
     *
     * @param object javabean
     * @return 返回json字符串
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static String toCamelCaseJsonString(Object object) {
        return JSON.toJSONString(object, SerializeConfigEnum.CamelCase.getSerializeConfig());
    }

    /**
     * javabean输出字符串key值为首字母大写驼峰
     *
     * @param object javabean
     * @return 返回json字符串
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static String toPascalCaseJsonString(Object object) {
        return JSON.toJSONString(object, SerializeConfigEnum.PascalCase.getSerializeConfig());
    }


    /**
     * javabean输出字符串key值为下划线
     *
     * @param object javabean
     * @return 返回json字符串
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static String toSnakeCaseJsonString(Object object) {
        return JSON.toJSONString(object, SerializeConfigEnum.SnakeCase.getSerializeConfig());
    }

    /**
     * javabean输出字符串key值为中划线格式
     *
     * @param object javabean
     * @return 返回json字符串
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static String toKebabCaseJsonString(Object object) {
        return JSON.toJSONString(object, SerializeConfigEnum.KebabCase.getSerializeConfig());
    }


    /**
     * 对象转换
     *
     * @param source 源对象
     * @param <T>    要转换成的对象类型
     * @return 转换后的对象
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static <T> T mapStruct(Object source, TypeReference<T> type) {
        if (source == null) {
            return null;
        }
        return JSON.parseObject(replace(JSON.toJSONString(source)), type);
    }

    /**
     * 对象转换
     *
     * @param source 源对象
     * @param <T>    要转换成的对象类型
     * @return 转换后的对象
     * @author Li Ruitong 86415270@qq.com
     * @since 1.0.0
     */
    public static <T> T mapStruct(Object source, Class<T> type) {
        if (source == null) {
            return null;
        }
        return JSON.parseObject(replace(JSON.toJSONString(source)), type);
    }

    /**
     * 替换字符串
     *
     * @param source 源字符串
     * @return 替换后字符串
     */
    public static String replace(String source) {
        return source.replace("\\", "")
            .replace("\"{", "{")
            .replace("}\"", "}")
            .replace("[\"", "[")
            .replace("]\"", "]");
    }

    /**
     * 序列化配置
     *
     * @author Li Ruitong 86415270@qq.com
     * @version 1.0
     * @since 1.0.0
     */
    private enum SerializeConfigEnum {
        /**
         * 代表一个实例
         */
        CamelCase(PropertyNamingStrategy.CamelCase),
        PascalCase(PropertyNamingStrategy.PascalCase),
        SnakeCase(PropertyNamingStrategy.SnakeCase),
        KebabCase(PropertyNamingStrategy.KebabCase),
        ;

        /**
         * 序列化配置
         */
        private SerializeConfig serializeConfig;

        /**
         * 反序列化配置
         */
        private ParserConfig parserConfig;

        /**
         * 构造方法-创建序列化配置对象并进行配置
         */
        SerializeConfigEnum(PropertyNamingStrategy propertyNamingStrategy) {
            serializeConfig = new SerializeConfig();
            serializeConfig.propertyNamingStrategy = propertyNamingStrategy;
            parserConfig = new ParserConfig();
            parserConfig.propertyNamingStrategy = propertyNamingStrategy;
        }

        /**
         * 获取序列化配置对象
         *
         * @return com.alibaba.fastjson.serializer.SerializeConfig 返回序列化配置对象
         * @author Li Ruitong 86415270@qq.com
         * @since 1.0.0
         */
        public SerializeConfig getSerializeConfig() {
            return serializeConfig;
        }

        /**
         * 获取反序列化配置对象
         *
         * @return com.alibaba.fastjson.parser.ParserConfig 返回反序列化配置对象
         * @author Li Ruitong 86415270@qq.com
         * @since 1.0.0
         */
        public ParserConfig getParserConfig() {
            return parserConfig;
        }
    }
}
