package cn.ziyicloud.framework.boot.autoconfigure.web.handler;

import cn.ziyicloud.framework.boot.enums.ErrorCodeEnum;
import cn.ziyicloud.framework.boot.exception.BaseException;
import cn.ziyicloud.framework.boot.model.BaseResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import java.util.List;

/**
 * 统一异常 处理
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.0
 */
@Slf4j
@RestControllerAdvice
@Order(1)
public class ZiyiExceptionHandler {
    /**
     * 拦截自定义异常进行处理
     *
     * @param e ZiyiBootException
     * @return ZiyiBootResponse
     */
    @ExceptionHandler(BaseException.class)
    public BaseResponse exceptionHandler(BaseException e) {
        debugLog(e);
        return new BaseResponse(e);
    }

    /**
     * post参数解析错误
     */
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public BaseResponse exceptionHandler(HttpMessageNotReadableException e) {
        debugLog(e);
        return new BaseResponse(new BaseException(ErrorCodeEnum.BAD_REQUEST, e.getMessage(), e));
    }

    /**
     * 请求方法不支持
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public BaseResponse exceptionHandler(HttpRequestMethodNotSupportedException e) {
        debugLog(e);
        return new BaseResponse(new BaseException(ErrorCodeEnum.METHOD_NOT_SUPPORTED, e));
    }

    /**
     * 请求Content type不支持
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public BaseResponse exceptionHandler(HttpMediaTypeNotSupportedException e) {
        debugLog(e);
        return new BaseResponse(new BaseException(ErrorCodeEnum.MEDIA_TYPE_NOT_SUPPORTED, e));
    }

    /**
     * 参数校验错误
     */
    @ExceptionHandler(BindException.class)
    public BaseResponse exceptionHandler(BindException e) {
        debugLog(e);
        List<ObjectError> objectErrors = e.getBindingResult().getAllErrors();
        return getValidExceptionResult(objectErrors);
    }

    /**
     * 参数校验错误
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResponse exceptionHandler(MethodArgumentNotValidException e) {
        debugLog(e);
        List<ObjectError> objectErrors = e.getBindingResult().getAllErrors();
        return getValidExceptionResult(objectErrors);
    }


    /**
     * 拦截Exception异常
     */
    @ExceptionHandler(Exception.class)
    public BaseResponse exceptionHandler(Exception e) {
        debugLog(e);
        return new BaseResponse(new BaseException(ErrorCodeEnum.BAD_REQUEST, e.getMessage(), e));
    }

    /**
     * 获取参数校验异常响应代码
     */
    private BaseResponse getValidExceptionResult(List<ObjectError> objectErrors) {
        StringBuilder sb = new StringBuilder();
        for (ObjectError error : objectErrors) {
            sb.append(error.getDefaultMessage()).append(";");
        }
        String message = sb.length() > 0 ? sb.toString().substring(0, sb.length() - 1) : sb.toString();
        return new BaseResponse(ErrorCodeEnum.PARAMETER_NOT_SUPPORTED, message);
    }

    /**
     * 记录日志
     */
    private void debugLog(Exception e) {
        if (log.isDebugEnabled()) {
            log.debug("响应异常", e);
        }
    }
}
