package cn.ziyicloud.framework.boot.util.httpclient;

import cn.ziyicloud.framework.boot.exception.ZiyiException;
import cn.ziyicloud.framework.boot.util.httpclient.builder.HttpClientBuilderUtil;
import cn.ziyicloud.framework.boot.util.httpclient.config.HttpConfig;
import cn.ziyicloud.framework.boot.util.httpclient.model.HttpMethod;
import cn.ziyicloud.framework.boot.util.httpclient.request.HttpDeleteWithBody;
import cn.ziyicloud.framework.boot.util.httpclient.util.HttpUrlUtils;
import cn.ziyicloud.framework.boot.util.httpclient.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.*;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static cn.ziyicloud.framework.boot.enums.ErrorCodeEnum.HTTP_ERROR;

/**
 * 使用HttpClient模拟发送（http/https）请求
 *
 * @author Li Ruitong
 */
@Slf4j
public class HttpClientUtil {
    private HttpClientUtil() {
    }

    /**
     * 请求超时时间
     */
    private static final int WAIT_TIMEOUT = 60 * 1000;
    /**
     * 最大连接数
     */
    public final static int MAX_TOTAL_CONNECTIONS = 800;
    /**
     * 每个路由最大连接数
     */
    public final static int MAX_ROUTE_CONNECTIONS = 400;
    /**
     * 连接超时时间
     */
    public final static int CONNECT_TIMEOUT = 10000;
    /**
     * 读取超时时间
     */
    public final static int READ_TIMEOUT = 10000;
    /**
     * 默认采用的http协议的HttpClient对象--开启连接池
     */
    private static HttpClient client4HTTP;

    /**
     * 默认采用的https协议的HttpClient对象-开启连接池
     */
    private static HttpClient client4HTTPS;

    static {
        try {
            client4HTTP = HttpClientBuilderUtil.custom()
                .pool(MAX_TOTAL_CONNECTIONS, MAX_ROUTE_CONNECTIONS)
                // .retry(3)
                .timeout(WAIT_TIMEOUT)
                .build();
            client4HTTPS = HttpClientBuilderUtil.custom()
                .pool(MAX_TOTAL_CONNECTIONS, MAX_ROUTE_CONNECTIONS)
                // .retry(3)
                .timeout(WAIT_TIMEOUT)
                .ssl().build();
        } catch (ZiyiException e) {
            log.debug("创建https协议的HttpClient对象出错", e);
        }
    }

    /**
     * 判定是否开启连接池、及url是http还是https <br>
     * 如果已开启连接池，则自动调用build方法，从连接池中获取client对象<br>
     * 否则，直接返回相应的默认client对象<br>
     *
     * @param config 请求参数配置
     */
    private static void create(HttpConfig config) {
        if (config.client() == null) {
            //如果为空，设为默认client对象
            if (config.url().toLowerCase().startsWith("https://")) {
                config.client(client4HTTPS);
            } else {
                config.client(client4HTTP);
            }
        }
    }


    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-------------------- 各请求方法 -------------------//

    /**
     * 以Get方式，请求资源或服务
     *
     * @param url 资源地址
     * @return 返回处理结果
     */
    public static String get(String url) {
        return get(HttpConfig.custom().url(url));
    }

    /**
     * 以Get方式，请求资源或服务
     *
     * @param url        资源地址
     * @param parameters 请求参数
     * @return 返回处理结果
     */
    public static String get(String url, Map<String, String> parameters) {
        StringBuilder stringBuffer = new StringBuilder();
        stringBuffer.append(url);
        if (HttpUrlUtils.checkHasParamters(url)) {
            stringBuffer.append("&");
        } else {
            stringBuffer.append("?");
        }
        stringBuffer.append(HttpUrlUtils.buildParameters(parameters));
        return get(HttpConfig.custom().url(stringBuffer.toString()));
    }

    /**
     * 以Get方式，请求资源或服务
     *
     * @param url     资源地址
     * @param headers 请求头信息
     * @return 返回处理结果
     */
    public static String get(String url, Header[] headers) {
        return get(HttpConfig.custom().url(url).headers(headers));
    }

    /**
     * 以Get方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String get(HttpClient client, String url, Header[] headers, HttpContext context, String encoding) {
        return get(HttpConfig.custom().client(client).url(url).headers(headers).context(context).encoding(encoding));
    }

    /**
     * 以Get方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回结果
     */
    public static String get(HttpConfig config) {
        return send(config.method(HttpMethod.GET));
    }

    /**
     * 以Post方式，请求资源或服务
     *
     * @param url       资源地址
     * @param jsonParam 请求参数
     * @param headers   请求头信息
     * @return 返回处理结果
     */
    public static String post(String url, String jsonParam, Header[] headers) {
        return post(HttpConfig.custom().url(url).json(jsonParam).headers(headers));
    }

    /**
     * 以Post方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param parasMap 请求参数
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String post(HttpClient client, String url, Header[] headers, Map<String, Object> parasMap, HttpContext context, String encoding) {
        return post(HttpConfig.custom().client(client).url(url).headers(headers).map(parasMap).context(context).encoding(encoding));
    }

    /**
     * 以Post方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String post(HttpConfig config) {
        return send(config.method(HttpMethod.POST));
    }

    /**
     * 以Put方式，请求资源或服务
     *
     * @param url       资源地址
     * @param jsonParam 请求参数
     * @param headers   请求头信息
     * @return 返回处理结果
     */
    public static String put(String url, String jsonParam, Header[] headers) {
        return put(HttpConfig.custom().url(url).json(jsonParam).headers(headers));
    }

    /**
     * 以Put方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param parasMap 请求参数
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String put(HttpClient client, String url, Map<String, Object> parasMap, Header[] headers, HttpContext context, String encoding) {
        return put(HttpConfig.custom().client(client).url(url).headers(headers).map(parasMap).context(context).encoding(encoding));
    }

    /**
     * 以Put方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String put(HttpConfig config) {
        return send(config.method(HttpMethod.PUT));
    }

    /**
     * 以Delete方式，请求资源或服务
     *
     * @param url     资源地址
     * @param headers 请求头信息
     * @return 返回处理结果
     */
    public static String delete(String url, Header[] headers) {
        return delete(HttpConfig.custom().url(url).headers(headers));
    }

    /**
     * 以Delete方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String delete(HttpClient client, String url, Header[] headers, HttpContext context, String encoding) {
        return delete(HttpConfig.custom().client(client).url(url).headers(headers).context(context).encoding(encoding));
    }

    /**
     * 以Delete方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String delete(HttpConfig config) {
        return send(config.method(HttpMethod.DELETE));
    }

    /**
     * 以Patch方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param parasMap 请求参数
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String patch(HttpClient client, String url, Map<String, Object> parasMap, Header[] headers, HttpContext context, String encoding) {
        return patch(HttpConfig.custom().client(client).url(url).headers(headers).map(parasMap).context(context).encoding(encoding));
    }

    /**
     * 以Patch方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String patch(HttpConfig config) {
        return send(config.method(HttpMethod.PATCH));
    }

    /**
     * 以Head方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String head(HttpClient client, String url, Header[] headers, HttpContext context, String encoding) {
        return head(HttpConfig.custom().client(client).url(url).headers(headers).context(context).encoding(encoding));
    }

    /**
     * 以Head方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String head(HttpConfig config) {
        return send(config.method(HttpMethod.HEAD));
    }

    /**
     * 以Options方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String options(HttpClient client, String url, Header[] headers, HttpContext context, String encoding) {
        return options(HttpConfig.custom().client(client).url(url).headers(headers).context(context).encoding(encoding));
    }

    /**
     * 以Options方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String options(HttpConfig config) {
        return send(config.method(HttpMethod.OPTIONS));
    }

    /**
     * 以Trace方式，请求资源或服务
     *
     * @param client   client对象
     * @param url      资源地址
     * @param headers  请求头信息
     * @param context  http上下文，用于cookie操作
     * @param encoding 编码
     * @return 返回处理结果
     */
    public static String trace(HttpClient client, String url, Header[] headers, HttpContext context, String encoding) {
        return trace(HttpConfig.custom().client(client).url(url).headers(headers).context(context).encoding(encoding));
    }


    /**
     * 以Trace方式，请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String trace(HttpConfig config) {
        return send(config.method(HttpMethod.TRACE));
    }

    /**
     * 下载文件
     *
     * @param url 资源地址
     * @return 返回处理结果
     */
    public static byte[] down(String url) {
        HttpConfig config = HttpConfig.custom().url(url).method(HttpMethod.GET);
        return format2bytes(execute(config));
    }

    /**
     * 下载文件
     *
     * @param url     资源地址
     * @param headers 请求头信息
     * @param out     输出流
     * @return 返回处理结果
     */
    public static OutputStream down(String url, OutputStream out, Header[] headers) {
        return down(HttpConfig.custom().headers(headers).out(out).url(url));
    }

    /**
     * 下载文件
     *
     * @param client  client对象
     * @param url     资源地址
     * @param headers 请求头信息
     * @param context http上下文，用于cookie操作
     * @param out     输出流
     * @return 返回处理结果
     */
    public static OutputStream down(HttpClient client, String url, Header[] headers, HttpContext context, OutputStream out) {
        return down(HttpConfig.custom().client(client).url(url).headers(headers).context(context).out(out));
    }

    /**
     * 下载文件
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static OutputStream down(HttpConfig config) {
        if (config.method() == null) {
            config.method(HttpMethod.GET);
        }
        return format2Stream(execute(config), config.out());
    }

    /**
     * 上传文件
     *
     * @param client  client对象
     * @param url     资源地址
     * @param headers 请求头信息
     * @param context http上下文，用于cookie操作
     * @return 返回处理结果
     */
    public static String upload(HttpClient client, String url, Header[] headers, HttpContext context) {
        return upload(HttpConfig.custom().client(client).url(url).headers(headers).context(context));
    }

    /**
     * 上传文件
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String upload(HttpConfig config) {
        if (config.method() != HttpMethod.POST && config.method() != HttpMethod.PUT) {
            config.method(HttpMethod.POST);
        }
        return send(config);
    }

    /**
     * 查看资源链接情况，返回状态码
     *
     * @param client  client对象
     * @param url     资源地址
     * @param headers 请求头信息
     * @param context http上下文，用于cookie操作
     * @return 返回处理结果
     */
    public static int status(HttpClient client, String url, Header[] headers, HttpContext context, HttpMethod method) {
        return status(HttpConfig.custom().client(client).url(url).headers(headers).context(context).method(method));
    }

    /**
     * 查看资源链接情况，返回状态码
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static int status(HttpConfig config) {
        return format2Int(execute(config));
    }

    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-------------------- 发送请求 --------------------//

    /**
     * 请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回处理结果
     */
    public static String send(HttpConfig config) {
        return format2String(execute(config), config.outenc());
    }

    /**
     * 请求资源或服务
     *
     * @param config 请求参数配置
     * @return 返回HttpResponse对象
     */
    private static HttpResponse execute(HttpConfig config) {
        //获取链接
        create(config);

        HttpResponse httpResponse;

        try {
            //创建请求对象
            HttpRequestBase request = getRequest(config.url(), config.method());

            //设置header信息
            request.setHeaders(config.headers());

            //判断是否支持设置entity(仅HttpPost、HttpPut、HttpPatch支持)
            if (HttpEntityEnclosingRequestBase.class.isAssignableFrom(request.getClass())) {
                List<NameValuePair> nvps = new ArrayList<>();

                //检测url中是否存在参数
                config.url(Utils.checkHasParas(config.url(), nvps, config.inenc()));

                //装填参数
                HttpEntity entity = Utils.map2HttpEntity(nvps, config.map(), config.inenc());

                //设置参数到请求对象中
                ((HttpEntityEnclosingRequestBase) request).setEntity(entity);

                log.debug("请求地址：" + config.url());
                if (nvps.size() > 0) {
                    log.debug("请求参数：" + nvps.toString());
                }
                if (config.json() != null) {
                    log.debug("请求参数：" + config.json());
                }
            } else {
                int idx = config.url().indexOf("?");
                log.debug("请求地址：" + config.url().substring(0, (idx > 0 ? idx : config.url().length())));
                if (idx > 0) {
                    log.debug("请求参数：" + config.url().substring(idx + 1));
                }
            }
            //执行请求操作，并拿到结果（同步阻塞）
            httpResponse = (config.context() == null) ? config.client().execute(request) : config.client().execute(request, config.context());

            if (config.isReturnRespHeaders()) {
                //获取所有response的header信息
                config.headers(httpResponse.getAllHeaders());
            }

            //获取结果实体
            return httpResponse;
        } catch (IOException e) {
            throw new ZiyiException(HTTP_ERROR, e);
        }
    }

    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //--------------------私有方法----------------------//

    /**
     * 转化为字符串
     *
     * @param httpResponse 响应对象
     * @param encoding     编码
     * @return 返回处理结果
     */
    private static String format2String(HttpResponse httpResponse, String encoding) {
        String body;
        try {
            if (httpResponse.getEntity() != null) {
                // 按指定编码转换结果实体为String类型
                body = EntityUtils.toString(httpResponse.getEntity(), encoding);
                log.debug(body);
            } else {
                //有可能是head请求
                body = httpResponse.getStatusLine().toString();
            }
            //关闭流
            EntityUtils.consume(httpResponse.getEntity());
        } catch (IOException e) {
            throw new ZiyiException(HTTP_ERROR, "http请求转化为字符串出错", e);
        } finally {
            close(httpResponse);
        }
        return body;
    }

    /**
     * 转化为数字
     *
     * @param httpResponse 响应对象
     * @return 返回处理结果
     */
    private static int format2Int(HttpResponse httpResponse) {
        int statusCode;
        try {
            statusCode = httpResponse.getStatusLine().getStatusCode();
            EntityUtils.consume(httpResponse.getEntity());
        } catch (IOException e) {
            throw new ZiyiException(HTTP_ERROR, e);
        } finally {
            close(httpResponse);
        }
        return statusCode;
    }

    /**
     * 转化为流
     *
     * @param httpResponse 响应对象
     * @param out          输出流
     * @return 返回输出流
     */
    public static OutputStream format2Stream(HttpResponse httpResponse, OutputStream out) {
        try {
            httpResponse.getEntity().writeTo(out);
            EntityUtils.consume(httpResponse.getEntity());
        } catch (IOException e) {
            throw new ZiyiException(HTTP_ERROR, e);
        } finally {
            close(httpResponse);
        }
        return out;
    }

    /**
     * 转换为字节数组
     *
     * @param httpResponse 响应对象
     * @return 返回字节数组
     */
    public static byte[] format2bytes(HttpResponse httpResponse) {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            InputStream inStream = httpResponse.getEntity().getContent();
            byte[] buffer = new byte[4096];
            int n;
            while (-1 != (n = inStream.read(buffer))) {
                output.write(buffer, 0, n);
            }
            EntityUtils.consume(httpResponse.getEntity());
        } catch (Exception e) {
            throw new ZiyiException(HTTP_ERROR, e);
        } finally {
            close(httpResponse);
        }
        return output.toByteArray();
    }

    /**
     * 根据请求方法名，获取request对象
     *
     * @param url    资源地址
     * @param method 请求方式
     * @return 返回Http处理request基类
     */
    private static HttpRequestBase getRequest(String url, HttpMethod method) {
        HttpRequestBase request;
        switch (method.getCode()) {
            case 0:
                // HttpGet
                request = new HttpGet(url);
                break;
            case 1:
                // HttpPost
                request = new HttpPost(url);
                break;
            case 2:
                // HttpHead
                request = new HttpHead(url);
                break;
            case 3:
                // HttpPut
                request = new HttpPut(url);
                break;
            case 4:
                // HttpDelete
                request = new HttpDeleteWithBody(url);
                break;
            case 5:
                // HttpTrace
                request = new HttpTrace(url);
                break;
            case 6:
                // HttpPatch
                request = new HttpPatch(url);
                break;
            case 7:
                // HttpOptions
                request = new HttpOptions(url);
                break;
            default:
                request = new HttpPost(url);
                break;
        }
        return request;
    }

    /**
     * 尝试关闭response
     *
     * @param httpResponse HttpResponse对象
     */
    private static void close(HttpResponse httpResponse) {
        try {
            if (httpResponse == null) {
                return;
            }
            //如果CloseableHttpResponse 是resp的父类，则支持关闭
            if (CloseableHttpResponse.class.isAssignableFrom(httpResponse.getClass())) {
                ((CloseableHttpResponse) httpResponse).close();
            }
        } catch (IOException e) {
            log.debug("尝试关闭response发生异常", e);
        }
    }
}
