package cn.ziyicloud.framework.boot.util.httpclient.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

/**
 * url构建工具类
 *
 * @author Li Ruitong
 */
@Slf4j
public class HttpUrlUtils {

    /**
     * 构建https无参数请求url
     *
     * @param hostname hostname
     * @param port     port
     * @param path     path
     * @return 返回url
     */
    public static String buildHttps(String hostname, String port, String path) {
        return build(Protocol.HTTPS, hostname, port, path, null);
    }

    /**
     * 构建https请求url
     *
     * @param hostname   hostname
     * @param port       port
     * @param path       path
     * @param parameters parameters
     * @return 返回url
     */
    public static String buildHttps(String hostname, String port, String path, Map<String, String> parameters) {
        return build(Protocol.HTTPS, hostname, port, path, parameters);
    }

    /**
     * 构建http无参数请求url
     *
     * @param hostname hostname
     * @param port     port
     * @param path     path
     * @return 返回url
     */
    public static String buildHttp(String hostname, String port, String path) {
        return build(Protocol.HTTP, hostname, port, path, null);
    }

    /**
     * 构建http请求url
     *
     * @param hostname   hostname
     * @param port       port
     * @param path       path
     * @param parameters parameters
     * @return 返回url
     */
    public static String buildHttp(String hostname, String port, String path, Map<String, String> parameters) {
        return build(Protocol.HTTP, hostname, port, path, parameters);
    }

    /**
     * 构建请求url
     *
     * @param protocol   协议
     * @param hostname   hostname
     * @param port       port
     * @param path       path
     * @param parameters parameters
     * @return 返回url
     */
    public static String build(Protocol protocol, String hostname, String port, String path
        , Map<String, String> parameters) {
        if (log.isDebugEnabled()) {
            log.debug("开始构建参数。protocol={};hostname={};port={},path={},parameters={}"
                , protocol.prefix, hostname, port, path, port);
        }
        StringBuilder url = new StringBuilder();
        //添加协议前缀和hostname
        url.append(protocol.getPrefix()).append(hostname);

        //端口不为空添加端口
        if (StringUtils.isNotEmpty(port)) {
            url.append(":").append(port);
        }

        //添加path
        String separate = "/";
        if (StringUtils.isNotEmpty(path)) {
            if (!path.startsWith(separate)) {
                url.append(separate);
            }
            url.append(path);
            if (path.endsWith(separate)) {
                url.deleteCharAt(url.length() - 1);
            }
        }

        //添加地址栏参数
        String urlParam = buildParameters(parameters);
        if (StringUtils.isNotEmpty(urlParam)) {
            if (checkHasParamters(path)) {
                url.append("&");
            } else {
                url.append("?");
            }
            url.append(urlParam);
        }

        return url.toString();
    }

    /**
     * 检测url是否含有参数
     *
     * @param url url
     * @return true 含有参数
     */
    public static boolean checkHasParamters(String url) {
        return url.contains("?") && url.indexOf("?") < url.indexOf("=");
    }

    /**
     * 构建地址栏参数
     *
     * @param parameters 参数列表
     * @return key=value&key2=value2
     */
    public static String buildParameters(Map<String, String> parameters) {
        if (parameters == null) {
            return "";
        }
        StringBuilder paramStr = new StringBuilder();
        //推荐使用通过Map.entrySet遍历key和value，尤其是容量大时
        for (Map.Entry<String, String> entry : parameters.entrySet()) {
            paramStr.append("&").append(entry.getKey()).append("=").append(entry.getValue());
        }
        if (paramStr.length() == 0) {
            return "";
        }
        return paramStr.substring(1);
    }

    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //-----------华----丽----分----割----线--------------//
    //----------------------- 内部类 -----------------------//

    /**
     * 协议版本 http/https
     *
     * @author Li Ruitong
     * @version 1.0
     * @date 2018/11/7 10:36
     */
    private enum Protocol {
        /**
         *
         */
        HTTP("http://"),
        /**
         *
         */
        HTTPS("https://");
        private String prefix;

        Protocol(String prefix) {
            this.prefix = prefix;
        }

        public String getPrefix() {
            return prefix;
        }
    }
}
