package cn.ziyicloud.framework.boot.util;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * id工具类
 *
 * @author Li Ruitong
 * @version 1.0
 * @date 2020/3/11
 */
public class IdUtils {
    /**
     * 从对象集合中获取对象的id列表
     *
     * @param col  要获取id列表的对象集合
     * @param <T>  对象类型
     * @param <ID> id类型
     * @return id列表
     */
    public static <T extends HasId<ID>, ID extends Serializable> Set<ID> getIds(Collection<T> col) {
        if (null == col) {
            return null;
        }
        return col.stream().filter(Objects::nonNull).map(T::getId).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    /**
     * 从对象集合中获取对象的id列表
     *
     * @param col  要获取id列表的对象集合
     * @param <T>  对象类型
     * @param <ID> id类型
     * @return id列表
     */
    public static <T extends HasPid<ID>, ID extends Serializable> Set<ID> getPids(Collection<T> col) {
        if (null == col) {
            return null;
        }
        return col.stream().filter(Objects::nonNull).map(T::getPid).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    /**
     * 对象集合转map
     *
     * @param col  要获取id列表的对象集合
     * @param <T>  对象类型
     * @param <ID> id类型
     * @return idMap
     */
    public static <T extends HasId<ID>, ID extends Serializable> Map<ID, T> getIdMap(Collection<T> col) {
        if (null == col) {
            return null;
        }
        return col.stream().collect(Collectors.toMap(T::getId,it->it,(v1,v2)->v1));
    }

    /**
     * 对象集合转map
     *
     * @param col  要获取id列表的对象集合
     * @param <T>  对象类型
     * @param <ID> id类型
     * @return idMap
     */
    public static <T extends HasPid<ID>, ID extends Serializable> Map<ID, T> getPidMap(Collection<T> col) {
        if (null == col) {
            return null;
        }
        return col.stream().collect(Collectors.toMap(T::getPid,it->it,(v1,v2)->v1));
    }

    /**
     * 想要用IdUtils工具简化获取id代码就要实现该接口
     *
     * @author Li Ruitong
     * @version 1.0
     * @date 2020/3/11
     */
    public interface HasId<ID extends Serializable> {
        /**
         * 获取id
         *
         * @return ID
         */
        ID getId();
    }

    /**
     * 想要用IdUtils工具简化获取pid代码就要实现该接口
     *
     * @author Li Ruitong
     * @version 1.0
     * @date 2020/3/11
     */
    public interface HasPid<ID extends Serializable> {
        /**
         * 获取id
         *
         * @return ID
         */
        ID getPid();
    }

    /**
     * 从对象集合中获取对象的id列表
     *
     * @param col  要获取id列表的对象集合
     * @param <T>  对象类型
     * @param <ID> id类型
     * @return id列表
     */
    public static <T extends HasId<ID>, ID extends Serializable> List<T> distinctById(Collection<T> col) {
        return col.stream().filter(distinctByKey(HasId::getId)).collect(Collectors.toList());
    }

    /**
     * 根据对象属性过滤
     *
     * @param keyExtractor keyExtractor
     * @param <T>          对象类型
     * @return true of false
     */
    private static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>(20);
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }
}
