package cn.ziyicloud.framework.boot.util.conlection;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 集合工具类
 *
 * @author Li Ruitong
 * @date 2020/5/14
 */
public class CollectionUtils {
    public CollectionUtils() {
    }

    private static final Logger log = LoggerFactory.getLogger(CollectionUtils.class);

    /**
     * 按每个集合中元素个数拆分list
     *
     * @param srcList 要拆分的集合
     * @param itemNum 每个集合元素个数
     * @param <T>     泛型
     * @return 拆分后的集合列表
     */
    public static <T> List<List<T>> splitListByItemNum(List<T> srcList, int itemNum) {
        List<List<T>> results = new ArrayList<>();
        if (srcList == null) {
            log.info("按每个集合中元素个数拆分list：srcList 不能为null");
            return results;
        }
        if (itemNum < 1) {
            log.info("按每个集合中元素个数拆分list：itemNum 不能小于1");
            return results;
        }
        //集合长度
        int size = srcList.size();
        //拆分个数
        int count = (int) Math.ceil(size * 1.0 / itemNum);
        for (int i = 0; i < count; i++) {
            int fromIndex = itemNum * i;
            int toIndex = Math.min(itemNum * (i + 1), size);
            //拆分
            results.add(srcList.subList(fromIndex, toIndex));
        }
        return results;
    }

    /**
     * 按个数拆分list
     *
     * @param srcList 要拆分的集合
     * @param listNum 要拆成集合个数
     * @param <T>     泛型
     * @return 拆分后的集合列表
     */
    public static <T> List<List<T>> splitList(List<T> srcList, int listNum) {
        List<List<T>> results = new ArrayList<>();
        if (srcList == null) {
            log.info("按个数拆分list：srcList 不能为null");
            return results;
        }
        if (listNum < 1) {
            log.info("按个数拆分list：itemNum 不能小于1");
            return results;
        }
        int size = srcList.size();
        if (size <= listNum) {
            return splitListByItemNum(srcList, 1);
        }
        //每个集合元素数
        int itemNum = size / listNum;
        for (int i = 0; i < listNum; i++) {
            int fromIndex = itemNum * i;
            int toIndex = itemNum * (i + 1);
            //最后一个集合分配剩余全部数据
            if (i == listNum - 1) {
                toIndex = srcList.size() - 1;
            }
            //拆分
            results.add(srcList.subList(fromIndex, toIndex));
        }
        return results;
    }

    /**
     * 集合分页
     *
     * @param srcList  要分页的集合
     * @param page     页码 从0开始
     * @param pageSize 每页记录数 大于1
     * @param <T>      泛型
     * @return 分页后的集合数据
     */
    public static <T> List<T> toPage(List<T> srcList, int page, int pageSize) {
        if (srcList == null) {
            log.info("集合分页：srcList 不能为null");
            return new ArrayList<>(0);
        }
        if (page < 0 || pageSize < 1) {
            log.info("集合分页：page不能小于0且pageSize不能小于1");
            return new ArrayList<>(0);
        }
        int fromIndex = pageSize * page;
        int toIndex = pageSize * (page + 1);
        int size = srcList.size();
        if (fromIndex > size) {
            return new ArrayList<>(0);
        } else if (toIndex >= size) {
            return srcList.subList(fromIndex, size);
        } else {
            return srcList.subList(fromIndex, toIndex);
        }
    }

    /**
     * 深度复制list集合--T实现序列化
     *
     * @param srcList 要深度复制的集合
     * @param <T>     泛型
     * @return List<T>返回复制后的list集合
     */
    @SuppressWarnings("unchecked")
    public static <T> List<T> deepCopyList(List<T> srcList) throws ClassNotFoundException {
        List<T> resultList = null;
        ByteArrayOutputStream bos = null;
        ObjectOutputStream os = null;
        ByteArrayInputStream bis = null;
        ObjectInputStream is = null;
        try {
            bos = new ByteArrayOutputStream();
            os = new ObjectOutputStream(bos);
            os.writeObject(srcList);
            bis = new ByteArrayInputStream(bos.toByteArray());
            is = new ObjectInputStream(bis);
            resultList = (List<T>) is.readObject();
        } catch (IOException e) {
            log.info("深度复制list集合出错", e);
        } finally {
            close(is);
            close(bis);
            close(os);
            close(bos);
        }

        return resultList;
    }

    /**
     * 求多个集合的交集
     *
     * @param elementLists List<List<T>>
     * @param <T>          泛型-重写equals方法
     * @return 返回交集
     */
    public static <T> List<T> retainElementList(List<List<T>> elementLists) {
        Optional<List<T>> result = elementLists.parallelStream()
            .filter(elementList -> elementList != null && (elementList).size() != 0)
            .reduce((a, b) -> {
                a.retainAll(b);
                return a;
            });
        return result.orElse(new ArrayList<>());
    }

    /**
     * 求A与B的差集。A-(A交B)
     *
     * @param colA A 集合
     * @param colB B集合
     * @param <T>  泛型-重写equals方法
     * @return 返回A与B的差集
     */
    public static <T> List<T> subtractions(Collection<T> colA, Collection<T> colB) {
        return colA.stream().filter(s -> !colB.contains(s)).collect(Collectors.toList());
    }

    /**
     * 求A与B的交集
     *
     * @param colA A 集合
     * @param colB B集合
     * @param <T>  泛型-重写equals方法
     * @return 返回交集
     */
    public static <T> List<T> intersections(Collection<T> colA, Collection<T> colB) {
        return colA.stream().filter(colB::contains).collect(Collectors.toList());
    }

    /**
     * 求A与B的并集
     *
     * @param colA A 集合
     * @param colB B集合
     * @param <T>  泛型-重写equals方法
     * @return 返回并集
     */
    public static <T> List<T> unions(Collection<T> colA, Collection<T> colB) {
        colA.addAll(colB);
        return colA.stream().distinct().collect(Collectors.toList());
    }

    /**
     * list 转为 set
     *
     * @param list list
     * @param <T>  <T>  泛型-重写equals方法
     * @return set
     */
    public static <T> Set<T> listToSet(List<T> list) {
        return new HashSet<>(list);
    }

    /**
     * set 转 list
     *
     * @param set set
     * @param <T> 泛型-重写equals方法
     * @return list
     */
    public static <T> List<T> setToList(Set<T> set) {
        return new ArrayList<>(set);
    }

    /**
     * 关闭资源流
     *
     * @param closeable 实现closeable接口的IO类
     */
    private static void close(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (IOException ex) {
                // ignore
            }
        }
    }
}
