package cn.ziyicloud.framework.boot.util.date;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Date;
import java.util.Objects;

/**
 * 日期处理工具类
 *
 * @author Li Ruitong 86415270@qq.com
 */
public class DateUtils {
    private DateUtils() {
    }

    /**
     * 时区
     */
    public static final ZoneId ZONE_ID = ZoneId.systemDefault();

    //日期字符串转换格式
    //-----------------------------------------------------------------------

    public static final DateTimeFormatter SIMPLE_DATE_PATTERN = DateTimeFormatter.ofPattern("yyyyMMdd");
    public static final DateTimeFormatter DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter TIME_PATTERN = DateTimeFormatter.ofPattern("HH:mm:ss");
    public static final DateTimeFormatter DATE_TIME_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    public static final DateTimeFormatter DATE_TIME_CN_PATTERN = DateTimeFormatter.ofPattern("yyyy年MM月dd日HH时mm分ss秒");
    public static final DateTimeFormatter MILLI_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS");
    public static final DateTimeFormatter FILE_STORE_DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy/MM/dd");

    /**
     * 获取日期格式化器
     *
     * @param pattern 模式
     * @return {@link DateTimeFormatter}
     */
    public static DateTimeFormatter getFormatter(String pattern) {
        return DateTimeFormatter.ofPattern(pattern);
    }

    //Date、LocalDate、LocalTime、LocalDateTime互转
    //-----------------------------------------------------------------------

    /**
     * Date转LocalDate
     *
     * @param date Date日期
     * @return LocalDate日期
     */
    public static LocalDate toLocalDate(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atZone(ZONE_ID).toLocalDate();
    }

    /**
     * Date转LocalTime
     *
     * @param date Date时间
     * @return LocalTime 时间
     */
    public static LocalTime toLocalTime(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atZone(ZONE_ID).toLocalTime();
    }

    /**
     * Date转LocalDateTime
     *
     * @param date Date时间
     * @return LocalDateTime 时间
     */
    public static LocalDateTime toLocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atZone(ZONE_ID).toLocalDateTime();
    }

    /**
     * Date转LocalDateTime
     *
     * @param timestamp 时间戳
     * @return LocalDateTime 时间
     */
    public static LocalDateTime toLocalDateTime(int timestamp) {
        return toLocalDateTime(new Date(timestamp * 1000L));
    }

    /**
     * localDate+LocalTime转LocalDateTime
     *
     * @param localDate LocalDate日期
     * @param localTime LocalTime时间
     * @return LocalDateTime 日期
     */
    public static LocalDateTime toLocalDateTime(LocalDate localDate, LocalTime localTime) {
        if (localDate == null) {
            localDate = LocalDate.now();
        }
        if (localTime == null) {
            localTime = LocalTime.now();
        }
        return localDate.atTime(localTime);
    }


    /**
     * LocalDate转Date日期
     *
     * @param localDate LocalDate日期
     * @return Date日期
     */
    public static Date toDate(LocalDate localDate) {
        if (localDate == null) {
            return null;
        }
        return Date.from(localDate.atStartOfDay().atZone(ZONE_ID).toInstant());
    }

    /**
     * 当天LocalTime 转Date日期
     *
     * @param localTime LocalTime时间
     * @return Date日期
     */
    public static Date toDate(LocalTime localTime) {
        return toDate(null, localTime);
    }

    /**
     * LocalDateTime转Date日期
     *
     * @param localDateTime LocalDateTime时间
     * @return Date日期
     */
    public static Date toDate(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
    }

    /**
     * LocalDate+LocalTime 转Date日期
     *
     * @param localDate LocalDate日期
     * @param localTime LocalTime时间
     * @return Date日期
     */
    public static Date toDate(LocalDate localDate, LocalTime localTime) {
        if (localDate == null) {
            localDate = LocalDate.now();
        }
        if (localTime == null) {
            localTime = LocalTime.MIN;
        }
        return Date.from(localDate.atTime(localTime).atZone(ZONE_ID).toInstant());
    }


    //LocalDateTime日期相关
    //-----------------------------------------------------------------------

    /**
     * 获取一天开始时间
     *
     * @param localDate LocalDate日期
     * @return LocalDateTime日期时间--当天00：00：00
     */
    public static LocalDateTime getDayStart(LocalDate localDate) {
        return toLocalDateTime(localDate, LocalTime.MIN);
    }

    /**
     * 获取一天结束时间
     *
     * @param localDate LocalDate日期
     * @return LocalDateTime日期时间--当天23：59：59.999999999
     */
    public static LocalDateTime getDayEnd(LocalDate localDate) {
        return toLocalDateTime(localDate, LocalTime.MAX);
    }

    //LocalDate日期相关
    //-----------------------------------------------------------------------

    /**
     * 获取某一天的日期
     *
     * @param offset 0今天，1明天，-1昨天，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getDay(int offset) {
        return LocalDate.now().plusDays(offset);
    }

    /**
     * 获取某周的开始日期
     *
     * @param offset 0本周，1下周，-1上周，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getWeekStart(int offset) {
        return LocalDate.now().plusWeeks(offset).with(DayOfWeek.MONDAY);
    }

    /**
     * 获取某周的结束日期
     *
     * @param offset 0本周，1下周，-1上周，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getWeekEnd(int offset) {
        return LocalDate.now().plusWeeks(offset).with(DayOfWeek.SUNDAY);
    }

    /**
     * 获取某月的开始日期
     *
     * @param offset 0本月，1下个月，-1上个月，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getMonthStart(int offset) {
        return LocalDate.now().plusMonths(offset).with(TemporalAdjusters.firstDayOfMonth());
    }

    /**
     * 获取某月的结束日期
     *
     * @param offset 0本月，1下个月，-1上个月，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getMonthEnd(int offset) {
        return LocalDate.now().plusMonths(offset).with(TemporalAdjusters.lastDayOfMonth());
    }

    /**
     * 获取某季度的开始日期
     * 季度一年四季， 第一季度：1月-3月， 第二季度：4月-6月， 第三季度：1月-9月， 第四季度：10月-12月
     *
     * @param offset 0本季度，1下个季度，-1上个季度，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getQuarterStart(int offset) {
        final LocalDate date = LocalDate.now().plusMonths(offset * 3L);
        //当月
        int month = date.getMonth().getValue();
        int start = getQuarterMonth(month, true);
        return date.plusMonths(start - month).with(TemporalAdjusters.firstDayOfMonth());
    }

    /**
     * 获取某季度的结束日期
     * 季度一年四季， 第一季度：1月-3月， 第二季度：4月-6月， 第三季度：1月-9月， 第四季度：10月-12月
     *
     * @param offset 0本季度，1下个季度，-1上个季度，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getQuarterEnd(int offset) {
        final LocalDate date = LocalDate.now().plusMonths(offset * 3L);
        //当月
        int month = date.getMonth().getValue();
        int end = getQuarterMonth(month, false);
        return date.plusMonths(end - month).with(TemporalAdjusters.lastDayOfMonth());
    }

    /**
     * 获取某年的开始日期
     *
     * @param offset 0本年，1明年，-1去年，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getYearStart(int offset) {
        return LocalDate.now().plusYears(offset).with(TemporalAdjusters.firstDayOfYear());
    }

    /**
     * 获取某年的结束日期
     *
     * @param offset 0本年，1明年，-1去年，依次类推
     * @return LocalDate 日期
     */
    public static LocalDate getYearEnd(int offset) {
        return LocalDate.now().plusYears(offset).with(TemporalAdjusters.lastDayOfYear());
    }

    /**
     * 获取某季度的开始或结果月份
     *
     * @param currentMonth 当前月
     * @param start        ture--要获取季度开始月份;false--要获取季度结束月份
     * @return int 季度开始或结束月份
     */
    public static int getQuarterMonth(int currentMonth, boolean start) {
        int result;
        switch (currentMonth) {
            case 1:
            case 2:
            case 3:
                result = start ? 1 : 3;
                break;
            case 4:
            case 5:
            case 6:
                result = start ? 4 : 6;
                break;
            case 7:
            case 8:
            case 9:
                result = start ? 7 : 9;
                break;
            case 10:
            case 11:
            case 12:
                result = start ? 10 : 12;
                break;
            default:
                result = currentMonth;
                break;
        }
        return result;
    }

    //时间戳相关
    //-----------------------------------------------------------------------

    /**
     * 获取当前unix时间戳--秒
     *
     * @param localDateTime LocalDateTime日期
     * @return int unix时间戳32位Integer
     */
    public static int toEpochSecond(LocalDateTime localDateTime) {
        return (int) localDateTime.atZone(ZONE_ID).toEpochSecond();
    }

    /**
     * 获取当前时间戳--毫秒
     *
     * @param localDateTime LocalDateTime日期
     * @return long 时间戳64位Long
     */
    public static long toEpochMilli(LocalDateTime localDateTime) {
        return localDateTime.atZone(ZONE_ID).toInstant().toEpochMilli();
    }

    /**
     * 获取当前unix时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int now() {
        return toEpochSecond(LocalDateTime.now());
    }

    /**
     * 获取今天开始时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int todayStart() {
        return toEpochSecond(getDayStart(LocalDate.now()));
    }

    /**
     * 获取今天结束时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int todayEnd() {
        return toEpochSecond(getDayEnd(LocalDate.now()));
    }

    /**
     * 获取本周结束时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int weekStart() {
        return toEpochSecond(getDayStart(getWeekStart(0)));
    }

    /**
     * 获取本月开始时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int weekEnd() {
        return toEpochSecond(getDayEnd(getWeekEnd(0)));
    }

    /**
     * 获取本周开始时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int monthStart() {
        return toEpochSecond(getDayStart(getMonthStart(0)));
    }

    /**
     * 获取本月结束时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int monthEnd() {
        return toEpochSecond(getDayEnd(getMonthEnd(0)));
    }

    /**
     * 获取本季度开始时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int quarterStart() {
        return toEpochSecond(getDayStart(getQuarterStart(0)));
    }

    /**
     * 获取本季度结束时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int quarterEnd() {
        return toEpochSecond(getDayEnd(getQuarterEnd(0)));
    }

    /**
     * 获取今年开始时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int yearStart() {
        return toEpochSecond(getDayStart(getYearStart(0)));
    }

    /**
     * 获取今年结束时间戳
     *
     * @return int unix时间戳32位Integer
     */
    public static int yearEnd() {
        return toEpochSecond(getDayEnd(getYearEnd(0)));
    }


    //字符串与日期转换
    //-----------------------------------------------------------------------

    /**
     * 解析yyyy-MM-dd格式的字符串,不含时分秒
     *
     * @param date yyyy-MM-dd格式的字符串
     * @return LocalDate日期
     */
    public static LocalDate parseDate(String date) {
        return LocalDate.parse(date, DATE_PATTERN);
    }

    /**
     * 转换成yyyy-MM-dd格式的字符串
     *
     * @param date LocalDate日期
     * @return yyyy-MM-dd格式的字符串
     */
    public static String formatDate(LocalDate date) {
        return date.format(DATE_PATTERN);
    }

    /**
     * 解析HH:mm:ss格式的字符串,不含年月日
     *
     * @param time HH:mm:ss格式的字符串
     * @return LocalTime 时间
     */
    public static LocalTime parseTime(String time) {
        return LocalTime.parse(time, TIME_PATTERN);
    }

    /**
     * 转换成HH:mm:ss格式的字符串
     *
     * @param time LocalTime时间
     * @return HH:mm:ss格式的时间
     */
    public static String formatTime(LocalTime time) {
        return time.format(TIME_PATTERN);
    }

    /**
     * 解析yyyy-MM-dd HH:mm:ss格式的字符串
     *
     * @param dateTime yyyy-MM-dd HH:mm:ss格式的字符串
     * @return LocalDateTime日期
     */
    public static LocalDateTime parseDateTime(String dateTime) {
        return LocalDateTime.parse(dateTime, DATE_TIME_PATTERN);
    }

    /**
     * 转换成yyyy-MM-dd HH:mm:ss格式的字符串
     *
     * @param dateTime LocalDateTime日期
     * @return yyyy-MM-dd HH:mm:ss格式的日期
     */
    public static String formatDateTime(LocalDateTime dateTime) {
        return dateTime.format(DATE_TIME_PATTERN);
    }


    //两个日期相差时间
    //-----------------------------------------------------------------------

    /**
     * 计算两个日期相差天数  1.1 1.2相差一天
     *
     * @param begin LocalDate 开始日期
     * @param end   LocalDate 结束日期
     * @return long 两个日期相差天数
     */
    public static long betweenDays(LocalDate begin, LocalDate end) {
        Objects.requireNonNull(begin, "begin");
        Objects.requireNonNull(end, "end");
        return end.toEpochDay() - begin.toEpochDay();
    }

    /**
     * 计算两个日期间隔几个月
     *
     * @param begin LocalDate 开始日期
     * @param end   LocalDate 结束日期
     * @return long 两个日期相差几个月
     */
    public static long betweenMonths(LocalDate begin, LocalDate end) {
        Objects.requireNonNull(begin, "begin");
        Objects.requireNonNull(end, "end");
        int beginYear = begin.getYear();
        int endYear = end.getYear();
        int startMonth = begin.getMonthValue();
        int endMonth = end.getMonthValue();
        return (endYear - beginYear) * 12L + endMonth - startMonth;
    }

    /**
     * 计算两个日期间隔几个月
     *
     * @param begin LocalDate 开始日期
     * @param end   LocalDate 结束日期
     * @return long 两个日期相差几个月
     */
    public static long betweenYears(LocalDate begin, LocalDate end) {
        Objects.requireNonNull(begin, "begin");
        Objects.requireNonNull(end, "end");
        return end.getYear() - begin.getYear();
    }

    /**
     * 获取今天时间为存储使用
     *
     * @return 今天时间字符串yyyy/MM/dd
     */
    public static String nowForFileStore() {
        return LocalDate.now().format(FILE_STORE_DATE_PATTERN);
    }
}
