package cn.ziyicloud.framework.boot.util.httpclient.builder;


import cn.ziyicloud.framework.boot.util.httpclient.config.SSLConfig;
import cn.ziyicloud.framework.boot.util.httpclient.model.SSLProtocolVersion;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.NoHttpResponseException;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.DefaultProxyRoutePlanner;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import java.io.InterruptedIOException;
import java.net.UnknownHostException;

/**
 * httpclient创建者
 *
 * @author Li Ruitong 86415270@qq.com
 */
public class HttpClientBuilderUtil extends HttpClientBuilder {
    /**
     * 记录是否设置了连接池
     */
    public boolean isSetPool = false;
    /**
     * ssl 协议版本
     */
    private SSLProtocolVersion sslpv = SSLProtocolVersion.TLSv1_2;

    private SSLConfig ssls = SSLConfig.getInstance();

    /**
     * 私有构造方法
     */
    private HttpClientBuilderUtil() {

    }

    /**
     * 获取实例
     *
     * @return 返回当前对象
     */
    public static HttpClientBuilderUtil custom() {
        return new HttpClientBuilderUtil();
    }

    /**
     * 设置超时时间
     *
     * @param timeout 超市时间，单位-毫秒
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil timeout(int timeout) {
        return timeout(timeout, true);
    }

    /**
     * 设置超时时间以及是否允许网页重定向（自动跳转 302）
     *
     * @param timeout        超时时间，单位-毫秒
     * @param redirectEnable 自动跳转
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil timeout(int timeout, boolean redirectEnable) {
        // 配置请求的超时设置
        RequestConfig config = RequestConfig.custom()
            .setConnectionRequestTimeout(timeout)
            .setConnectTimeout(timeout)
            .setSocketTimeout(timeout)
            .setRedirectsEnabled(redirectEnable)
            .build();
        return (HttpClientBuilderUtil) this.setDefaultRequestConfig(config);
    }

    /**
     * 设置ssl安全链接
     *
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil ssl() {
        return (HttpClientBuilderUtil) this.setSSLSocketFactory(ssls.getSSLCONNSF(sslpv));
    }

    /**
     * 设置自定义sslcontext
     *
     * @param keyStorePath 密钥库路径
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil ssl(String keyStorePath) {
        return ssl(keyStorePath, "nopassword");
    }

    /**
     * 设置自定义sslcontext
     *
     * @param keyStorePath 密钥库路径
     * @param keyStorepass 密钥库密码
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil ssl(String keyStorePath, String keyStorepass) {
        this.ssls = SSLConfig.custom().customSSL(keyStorePath, keyStorepass);
        return ssl();
    }


    /**
     * 设置连接池（默认开启https）
     *
     * @param maxTotal           最大连接数
     * @param defaultMaxPerRoute 每个路由默认连接数
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil pool(int maxTotal, int defaultMaxPerRoute) {
        Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder
            .<ConnectionSocketFactory>create()
            .register("http", PlainConnectionSocketFactory.INSTANCE)
            .register("https", ssls.getSSLCONNSF(sslpv)).build();
        //设置连接池大小
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
        // 设置连接池最大数量,这个参数表示所有连接最大数。
        connManager.setMaxTotal(maxTotal);
        // 设置单个路由最大连接数量，表示单个域名的最大连接数，
        connManager.setDefaultMaxPerRoute(defaultMaxPerRoute);
        isSetPool = true;
        return (HttpClientBuilderUtil) this.setConnectionManager(connManager);
    }

    /**
     * 设置digest摘要认证
     *
     * @param username 用户名
     * @param password 密码
     * @return 当前对象
     */
    public HttpClientBuilderUtil digest(String username, String password) {
        return digest(username, password, null, null);
    }

    /**
     * 设置digest摘要认证
     *
     * @param username 用户名
     * @param password 密码
     * @param host     hostname
     * @param port     port
     * @return 当前对象
     */
    public HttpClientBuilderUtil digest(String username, String password, String host, Integer port) {
        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(new AuthScope(StringUtils.isBlank(host) ? AuthScope.ANY_HOST : host, port == null ? AuthScope.ANY_PORT : port),
            new UsernamePasswordCredentials(username, password));
        return (HttpClientBuilderUtil) this.setDefaultCredentialsProvider(credentialsProvider);
    }

    /**
     * 设置代理
     *
     * @param hostOrIP 代理host或者ip
     * @param port     代理端口
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil proxy(String hostOrIP, int port) {
        // 依次是代理地址，代理端口号，协议类型
        HttpHost proxy = new HttpHost(hostOrIP, port, "http");
        DefaultProxyRoutePlanner routePlanner = new DefaultProxyRoutePlanner(proxy);
        return (HttpClientBuilderUtil) this.setRoutePlanner(routePlanner);
    }

    /**
     * 重试（如果请求是幂等的，就再次尝试）
     *
     * @param tryTimes 重试次数
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil retry(final int tryTimes) {
        return retry(tryTimes, false);
    }

    /**
     * 重试（如果请求是幂等的，就再次尝试）
     *
     * @param tryTimes               重试次数
     * @param retryWhenInterruptedIo 连接拒绝时，是否重试
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil retry(final int tryTimes, final boolean retryWhenInterruptedIo) {
        // 请求重试处理
        HttpRequestRetryHandler httpRequestRetryHandler = (exception, executionCount, context) -> {
            if (executionCount >= tryTimes) {
                // 如果已经重试了n次，就放弃
                return false;
            }
            if (exception instanceof NoHttpResponseException) {
                // 如果服务器丢掉了连接，那么就重试
                return true;
            }
            if (exception instanceof ConnectTimeoutException) {
                //连接超时重试
                return true;
            }
            if (exception instanceof UnknownHostException) {
                // 目标服务器不可达
                return true;
            }

            if (exception instanceof SSLHandshakeException) {
                // 不要重试SSL握手异常
                return false;
            }
            if (exception instanceof SSLException) {
                // SSL握手异常
                return false;
            }
            if (exception instanceof InterruptedIOException) {
                // 连接拒绝时，是否重试
                return retryWhenInterruptedIo;
            }

            HttpClientContext clientContext = HttpClientContext.adapt(context);
            HttpRequest request = clientContext.getRequest();
            // 如果请求是幂等的，就再次尝试
            return !(request instanceof HttpEntityEnclosingRequest);
        };
        this.setRetryHandler(httpRequestRetryHandler);
        return this;
    }

    /**
     * 设置ssl版本<br>
     * 如果您想要设置ssl版本，必须<b><span style="color:red">先调用此方法，再调用ssl方法</span><br>
     * 仅支持 SSLv3，TSLv1，TSLv1.1，TSLv1.2</b>
     *
     * @param sslpv 版本号
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil sslpv(String sslpv) {
        return sslpv(SSLProtocolVersion.find(sslpv));
    }

    /**
     * 设置ssl版本<br>
     * 如果您想要设置ssl版本，必须<b>先调用此方法，再调用ssl方法<br>
     * 仅支持 SSLv3，TSLv1，TSLv1.1，TSLv1.2</b>
     *
     * @param sslpv 版本号
     * @return 返回当前对象
     */
    public HttpClientBuilderUtil sslpv(SSLProtocolVersion sslpv) {
        this.sslpv = sslpv;
        return this;
    }
}
