package cn.ziyicloud.framework.boot.util.json;

import cn.ziyicloud.framework.boot.enums.ErrorCodeEnum;
import cn.ziyicloud.framework.boot.exception.BaseException;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.text.SimpleDateFormat;
import java.util.List;

/**
 * using read and write json by jackson
 *
 * @author Li Ruitong 86415270@qq.com
 * @since 1.0.0
 */
@Slf4j
public class JsonUtils {
    private JsonUtils() {
    }

    private final static ObjectMapper JSON_OBJECT_MAPPER = new ObjectMapper();

    static {
        // 对象的所有字段全部列入，还是其他的选项，可以忽略null等
        JSON_OBJECT_MAPPER.setSerializationInclusion(JsonInclude.Include.ALWAYS);
        // 设置Date类型的序列化及反序列化格式
        JSON_OBJECT_MAPPER.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
        // 忽略空Bean转json的错误
        JSON_OBJECT_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        // 忽略未知属性，防止json字符串中存在，java对象中不存在对应属性的情况出现错误
        JSON_OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        // 注册一个时间序列化及反序列化的处理模块，用于解决jdk8中localDateTime等的序列化问题
        JSON_OBJECT_MAPPER.registerModule(new JavaTimeModule());
        //格式化输出
        JSON_OBJECT_MAPPER.configure(SerializationFeature.INDENT_OUTPUT, true);
    }

    /**
     * 获取json转换对象
     *
     * @return objectMapper
     */
    public static ObjectMapper getInstance() {
        return JSON_OBJECT_MAPPER;
    }

    /**
     * 转换为 JSON 字符串
     *
     * @param value 要转换的值
     * @return JSON 字符串
     */
    public static String toJson(Object value) {
        return toJson(value, JSON_OBJECT_MAPPER);
    }

    /**
     * 转换为 JSON 字符串，忽略空值
     *
     * @param value 要转换的值
     * @return JSON 字符串
     */
    public static String toJsonIgnoreNull(Object value) {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        return toJson(value, mapper);
    }

    /**
     * 转换为 JSON 字符串
     *
     * @param value  要转换的值
     * @param mapper ObjectMapper
     * @return JSON 字符串
     */
    public static String toJson(Object value, ObjectMapper mapper) {
        try {
            return mapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new BaseException(ErrorCodeEnum.UTIL_ERROR, e);
        }
    }

    /**
     * json字符串转为java类
     *
     * @param content   json字符串
     * @param valueType 要转换的对象类型
     * @param <T>       泛型
     * @return java 对象
     */
    public static <T> T parse(String content, Class<T> valueType) {
        return parse(content, valueType, null, JSON_OBJECT_MAPPER);
    }

    /**
     * json字符串转为java类
     *
     * @param content   json字符串
     * @param valueType 要转换的对象类型
     * @param mapper    json ObjectMapper
     * @param <T>       泛型
     * @return java 对象
     */
    public static <T> T parse(String content, Class<T> valueType, ObjectMapper mapper) {
        return parse(content, valueType, null, mapper);
    }

    /**
     * json字符串转为java类
     *
     * @param content   json字符串
     * @param valueType 要转换的对象类型
     * @param <T>       泛型
     * @return java 对象
     */
    public static <T> List<T> parseArray(String content, Class<T> valueType) {
        return parse(content, null, new TypeReference<List<T>>() {
        }, JSON_OBJECT_MAPPER);
    }

    /**
     * json字符串转为java类
     *
     * @param content   json字符串
     * @param valueType 要转换的对象类型
     * @param mapper    json ObjectMapper
     * @param <T>       泛型
     * @return java 对象
     */
    public static <T> List<T> parseArray(String content, Class<T> valueType, ObjectMapper mapper) {
        return parse(content, null, new TypeReference<List<T>>() {
        }, mapper);
    }

    /**
     * json字符串转为java类
     *
     * @param content      json字符串
     * @param valueTypeRef 要转换的对象
     * @param <T>          泛型
     * @return java 对象
     */
    public static <T> T parse(String content, TypeReference<T> valueTypeRef) {
        return parse(content, null, valueTypeRef, JSON_OBJECT_MAPPER);
    }

    /**
     * json字符串转为java类
     *
     * @param content      json字符串
     * @param valueTypeRef 要转换的对象
     * @param mapper       json ObjectMapper
     * @param <T>          泛型
     * @return java 对象
     */
    public static <T> T parse(String content, TypeReference<T> valueTypeRef, ObjectMapper mapper) {
        return parse(content, null, valueTypeRef, mapper);
    }

    /**
     * json字符串转为java类
     *
     * @param content      json字符串
     * @param valueType    要转换的对象类型
     * @param valueTypeRef 要转换的对象
     * @param mapper       json ObjectMapper
     * @param <T>          泛型
     * @return java 对象
     */
    public static <T> T parse(String content, Class<T> valueType, TypeReference<T> valueTypeRef, ObjectMapper mapper) {
        T obj = null;
        if (StringUtils.isNotEmpty(content)) {
            try {
                if (valueType != null) {
                    obj = mapper.readValue(content, valueType);
                } else {
                    obj = mapper.readValue(content, valueTypeRef);
                }
            } catch (JsonProcessingException e) {
                throw new BaseException(ErrorCodeEnum.UTIL_ERROR, e);
            }
        }
        return obj;
    }
}
