package cn.zzq0324.radish.components.docs;

import cn.zzq0324.radish.common.constant.ProfileConstants;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.v3.core.jackson.ModelResolver;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.SpecVersion;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

/**
 * 基于openapi的配置
 *
 * @author  zzq0324
 * @since : 1.0.0
 */
@Slf4j
@Configuration
@Profile({ProfileConstants.DEV, ProfileConstants.FAT})
public class OpenAPIConfiguration {

  @Autowired
  private Environment properties;
  
  @Bean
  @ConfigurationProperties("radish.doc")
  public DocProperties docProperties() {
    return new DocProperties();
  }

  @Bean
  public ModelResolver modelResolver(@Autowired(required = false) ObjectMapper objectMapper) {
    return new ModelResolver(objectMapper != null ? objectMapper : new ObjectMapper());
  }

  @Bean
  public OpenAPI openAPI(DocProperties docProperties) {
    OpenAPI openAPI = new OpenAPI(SpecVersion.V30).info(apiInfo(docProperties));

    // 设置认证信息
    if (StringUtils.hasLength(docProperties.getAuthHeader())) {
      openAPI.components(securityComponents(docProperties));
    }

    return openAPI;
  }

  private Info apiInfo(DocProperties docProperties) {
    Info info = new Info();
    info.setTitle(docProperties.getTitle());
    info.setDescription(docProperties.getDescription());
    info.setVersion(docProperties.getVersion());

    // 设置联系人信息
    if (docProperties.getContact() != null) {
      Contact contact = new Contact();
      contact.setName(docProperties.getContact().getName());
      contact.setUrl(docProperties.getContact().getUrl());
      contact.setEmail(docProperties.getContact().getEmail());
      info.setContact(contact);
    }

    return info;
  }

  private Components securityComponents(DocProperties docProperties) {
    Components components = new Components();
    Set<String> headers = StringUtils.commaDelimitedListToSet(docProperties.getAuthHeader());
    for (String header : headers) {
      components.addSecuritySchemes(header,
          new SecurityScheme().type(SecurityScheme.Type.APIKEY).description(header).in(SecurityScheme.In.HEADER)
              .name(header));
    }

    return components;
  }
}
