package cn.zzq0324.radish.components.http.configuration;

import cn.zzq0324.radish.common.spring.SpringContextHolder;
import cn.zzq0324.radish.components.http.RadishHttpClient;
import cn.zzq0324.radish.components.http.configuration.OkHttpProperties.ProxyProperties;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.util.StringUtils;

/**
 * OkHttp配置类
 *
 * @author: zzq0324
 * @since : 1.0.0
 */
@Slf4j
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties(OkHttpProperties.class)
public class OkHttpAutoConfiguration {

  private okhttp3.OkHttpClient okHttpClient;

  @Bean
  public RadishHttpClient radishHttpClient(OkHttpClient okHttpClient) {
    return new RadishHttpClient(okHttpClient);
  }

  @Bean
  public OkHttpClientFactory okHttpClientFactory() {
    return new OkHttpClientFactory();
  }

  @Bean
  public ConnectionPool httpClientConnectionPool(OkHttpProperties okHttpProperties) {
    int maxTotalConnections = okHttpProperties.getMaxConnections();
    long timeToLive = okHttpProperties.getTimeToLive();

    return new ConnectionPool(maxTotalConnections, timeToLive, TimeUnit.MILLISECONDS);
  }

  @Bean
  @DependsOn(SpringContextHolder.BEAN_NAME)
  public okhttp3.OkHttpClient okHttpClient(OkHttpClientFactory clientFactory, ConnectionPool connectionPool,
      OkHttpProperties okHttpProperties, Optional<List<Interceptor>> optionalInterceptors) {
    OkHttpClient.Builder builder = clientFactory.createBuilder(okHttpProperties.isDisableSslValidation());

    // 设置拦截器
    setInterceptors(builder, optionalInterceptors);

    // 设置代理
    setProxy(builder, okHttpProperties.getProxy());

    this.okHttpClient = builder.connectTimeout(okHttpProperties.getConnectionTimeout(), TimeUnit.MILLISECONDS)
        .followRedirects(okHttpProperties.isFollowRedirects())
        .readTimeout(Duration.ofMillis(okHttpProperties.getReadTimeout())).connectionPool(connectionPool)
        .retryOnConnectionFailure(okHttpProperties.isRetryOnConnectionFailure()).build();

    return this.okHttpClient;
  }

  /**
   * 设置拦截器，只要实现了OkHttpInterceptor就会扫描使用，实现类需要注解上@Extention
   */
  private void setInterceptors(OkHttpClient.Builder builder, Optional<List<Interceptor>> optionalInterceptors) {
    if (optionalInterceptors.isPresent()) {
      optionalInterceptors.get().stream().forEach(interceptor -> {
        log.info("set interceptor: {} to OkHttpClient", interceptor.getClass().getName());
        builder.addInterceptor(interceptor);
      });
    }
  }

  /**
   * 设置代理，如果存在一个Proxy的bean，则自动设置为代理
   */
  private void setProxy(OkHttpClient.Builder builder, ProxyProperties proxyProperties) {
    if (proxyProperties == null || !StringUtils.hasLength(proxyProperties.getHost())) {
      return;
    }

    Proxy.Type proxyType = Proxy.Type.valueOf(proxyProperties.getType().toUpperCase());
    InetSocketAddress address = new InetSocketAddress(proxyProperties.getHost(), proxyProperties.getPort());
    Proxy proxy = new Proxy(proxyType, address);

    builder.proxy(proxy);
  }

  @PreDestroy
  public void destroy() {
    if (this.okHttpClient != null) {
      this.okHttpClient.dispatcher().executorService().shutdown();
      this.okHttpClient.connectionPool().evictAll();
    }
  }
}
