package cn.zzq0324.radish.components.http.util;

import cn.zzq0324.radish.common.code.CommonStatusCode;
import cn.zzq0324.radish.common.exception.BusinessException;
import cn.zzq0324.radish.common.util.JsonUtils;
import cn.zzq0324.radish.components.http.constant.HttpMethodType;
import cn.zzq0324.radish.components.http.constant.MediaTypes;
import java.util.Map;
import java.util.Objects;
import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.RequestBody;
import okio.Buffer;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * 请求工具类
 *
 * @author zzq0324
 * @since : 1.0.0
 */
public class RequestUtils {

  /**
   * 是否包含指定的头部值
   */
  public static boolean containsHeaderValue(Request request, String headerName, String headerValue) {
    String value = request.header(headerName);

    return Objects.equals(value, headerValue);
  }

  /**
   * 是否包含指定的头部值
   */
  public static boolean containsHeader(Request request, String headerName) {
    String value = request.header(headerName);

    return StringUtils.hasLength(value);
  }

  public static Request buildRequest(String url, HttpMethodType methodType, Object requestData) {
    switch (methodType) {
      case GET:
        return buildGetRequest(url, requestData);
      case POST_FORM:
        return buildPostRequest(url, requestData);
      case POST_APPLICATION:
        return buildPostEntityRequest(url, requestData);

      default:
        throw new BusinessException(CommonStatusCode.METHOD_NOT_SUPPORT);
    }
  }

  /**
   * 构建get请求
   */
  public static Request buildGetRequest(String url) {
    return buildGetRequest(url, null);
  }

  /**
   * 构建get请求
   */
  public static Request buildGetRequest(String url, Object requestData) {
    HttpUrl.Builder httpBuilder = HttpUrl.parse(url).newBuilder();
    Map<String, Object> requestDataMap = objectToMap(requestData);

    if (!CollectionUtils.isEmpty(requestDataMap)) {
      requestDataMap.keySet().forEach(key -> {
        Object value = requestDataMap.get(key);
        httpBuilder.addQueryParameter(key, value == null ? null : value.toString());
      });
    }

    return new Request.Builder().url(httpBuilder.build()).build();
  }

  /**
   * post提交，content-type为application/json
   */
  public static Request buildPostEntityRequest(String url) {
    return buildPostEntityRequest(url, null);
  }

  /**
   * post提交，content-type为application/json
   */
  public static Request buildPostEntityRequest(String url, Object requestData) {
    RequestBody requestBody = RequestBody.create(JsonUtils.toJson(requestData), MediaTypes.APPLICATION_JSON);
    return new Request.Builder().url(url).post(requestBody).build();
  }

  public Request buildPostRequest(String url) {
    return buildPostRequest(url, null);
  }

  /**
   * post表单提交
   */
  public static Request buildPostRequest(String url, Object requestData) {
    FormBody.Builder formBuilder = new FormBody.Builder();
    Map<String, Object> requestDataMap = objectToMap(requestData);

    if (!CollectionUtils.isEmpty(requestDataMap)) {
      requestDataMap.keySet().forEach(key -> {
        Object value = requestDataMap.get(key);
        formBuilder.add(key, value == null ? null : value.toString());
      });
    }

    RequestBody requestForm = formBuilder.build();
    return new Request.Builder().url(url).post(requestForm).build();
  }

  private static Map<String, Object> objectToMap(Object data) {
    if (data == null) {
      return null;
    }

    if (data instanceof Map) {
      return (Map<String, Object>) data;
    }

    return JsonUtils.objectMapper().convertValue(data, Map.class);
  }

  public static String getUrl(Request request) {
    return request.url().url().toString();
  }

  public static String getBodyAsString(Request request) {
    if (request.body() == null) {
      return null;
    }

    try (Buffer buffer = new Buffer()) {
      try {
        request.body().writeTo(buffer);
        return buffer.readUtf8();
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
  }
}
