package cn.zzq0324.radish.components.jwt;

import cn.zzq0324.radish.common.util.JsonUtils;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTCreator.Builder;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.Claim;
import com.auth0.jwt.interfaces.DecodedJWT;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

/**
 * JWT工具类
 *
 * @author zhengzh
 * @since 1.0.0
 */
@Service
public class JwtUtils {

  public static final String CLAIM_NAME = "radishClaim";
  private static JwtProperties jwtProperties;
  private static Algorithm algorithm;

  /**
   * 通过构造器注入并初始化静态变量
   */
  public JwtUtils(@Autowired JwtProperties jwtProperties) {
    Assert.notNull(jwtProperties, "jwt properties required, please check if radish.jwt configured");
    Assert.notNull(jwtProperties.getSecret(), "jwt secret required.");

    JwtUtils.jwtProperties = jwtProperties;
    algorithm = Algorithm.HMAC256(jwtProperties.getSecret());
  }

  /**
   * 生成令牌，由于claim value不支持Object，因此直接存储在固定字段上
   */
  public static String generateToken(Map<String, Object> claims) {
    return generateToken(claims, jwtProperties.getExpiresIn(), TimeUnit.SECONDS);
  }

  /**
   * 生成令牌，由于claim value不支持Object，因此直接存储在固定字段上
   */
  public static String generateToken(Map<String, Object> claims, int expire, TimeUnit timeUnit) {
    Date expireDate = getExpireDate(expire, timeUnit);

    Builder builder = JWT.create().withIssuer(jwtProperties.getIssuer()).withExpiresAt(expireDate);
    if (!CollectionUtils.isEmpty(claims)) {
      builder.withClaim(CLAIM_NAME, claims);
    }

    return builder.sign(algorithm);
  }

  /**
   * 校验token
   */
  public static Map<String, Object> verifyToken(String token) {
    DecodedJWT decodedJWT = JWT.require(algorithm).withIssuer(jwtProperties.getIssuer()).build().verify(token);

    Claim claim = decodedJWT.getClaim(CLAIM_NAME);
    if (claim != null) {
      return claim.asMap();
    }

    return null;
  }

  /**
   * 校验token，并将claim中的信息转为对象返回
   */
  public static <T> T verifyToken(String token, Class<T> type) {
    Map<String, Object> claims = verifyToken(token);

    return JsonUtils.objectMapper().convertValue(claims, type);
  }

  /**
   * 刷新token，只有token校验通过才能刷新token
   */
  public static String refreshToken(String token) {
    Map<String, Object> claims = verifyToken(token);

    return generateToken(claims);
  }

  private static Date getExpireDate(int expire, TimeUnit timeUnit) {
    long mills = System.currentTimeMillis() + timeUnit.toMillis(expire);

    return new Date(mills);
  }
}
