package cn.zzq0324.radish.components.wechat.common;

import cn.zzq0324.radish.common.code.CommonStatusCode;
import cn.zzq0324.radish.common.exception.BusinessException;
import cn.zzq0324.radish.common.exception.InternalErrorException;
import cn.zzq0324.radish.common.spring.SpringContextHolder;
import cn.zzq0324.radish.common.util.JsonUtils;
import cn.zzq0324.radish.components.http.RadishHttpClient;
import cn.zzq0324.radish.components.http.constant.HttpMethodType;
import cn.zzq0324.radish.components.http.util.RequestUtils;
import cn.zzq0324.radish.components.wechat.dto.AppInfo;
import cn.zzq0324.radish.components.wechat.dto.CommonResponse;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;
import okhttp3.Request;
import org.springframework.util.StringUtils;

/**
 * 基础API
 *
 * @author: zzq0324
 * @since : 1.0.0
 */
@Slf4j
public class BaseApi {

  private static final String PARAMETER_ACCESS_TOKEN = "access_token";

  // 调用成功响应码，为0代表成功(null也代表成功)
  private static final String SUCCESS_CODE = "0";

  private static RadishHttpClient radishHttpClient = SpringContextHolder.getBean(RadishHttpClient.class);

  protected AppInfo appInfo;

  public BaseApi(AppInfo appInfo) {
    this.appInfo = appInfo;
  }

  protected <T extends CommonResponse> T execute(String url, HttpMethodType methodType, Object requestData,
      Class<T> responseType) {
    return execute(url, methodType, requestData, responseType, true);
  }

  protected <T extends CommonResponse> T execute(String url, HttpMethodType methodType, Object requestData,
      Class<T> responseType, boolean appendAccessToken) {
    Request request = RequestUtils.buildRequest(url, methodType, requestData);

    // 拼接Access Token
    if (appendAccessToken) {
      String accessToken = AccessTokenHolder.get(appInfo).getToken();
      HttpUrl httpUrl = request.url().newBuilder().addQueryParameter(PARAMETER_ACCESS_TOKEN, accessToken).build();
      request = request.newBuilder().url(httpUrl).build();
    }

    // 执行调用
    T response = radishHttpClient.call(request, responseType);
    if (!StringUtils.hasLength(response.getCode()) || SUCCESS_CODE.equals(response.getCode())) {
      return response;
    }

    log.error("invoke wechat api error, requestUrl: {}, requestData: {}, response: {}",
        new Object[]{request.url().url(), RequestUtils.getBodyAsString(request), JsonUtils.toJson(response)});

    throw new BusinessException(CommonStatusCode.THIRD_ERROR.args("invoke wechat error: " + response.getMessage()));
  }

  public static <T extends BaseApi> T newInstance(AppInfo appInfo, Class<T> type) {
    try {
      return (T) type.getDeclaredConstructor(AppInfo.class).newInstance(appInfo);
    } catch (Exception e) {
      throw new InternalErrorException("new api: " + type.getName() + " instance error", e);
    }
  }
}
