package cn.zzq0324.radish.components.wechat.controller;

import cn.zzq0324.radish.common.util.JsonUtils;
import cn.zzq0324.radish.common.util.XmlUtils;
import cn.zzq0324.radish.components.wechat.config.WechatAppProperties;
import cn.zzq0324.radish.components.wechat.constant.MsgType;
import cn.zzq0324.radish.components.wechat.crypto.MessageCrypto;
import cn.zzq0324.radish.components.wechat.dto.AppInfo;
import cn.zzq0324.radish.components.wechat.handler.WechatCallbackHandler;
import cn.zzq0324.radish.components.wechat.officialaccount.dto.calllback.AesEncryptData;
import cn.zzq0324.radish.components.wechat.officialaccount.dto.calllback.DecryptedData;
import cn.zzq0324.radish.components.wechat.officialaccount.dto.calllback.EventCallbackRequest;
import cn.zzq0324.radish.components.wechat.officialaccount.dto.calllback.VerifyUrlRequest;
import cn.zzq0324.radish.extension.BusinessScenario;
import cn.zzq0324.radish.extension.ExtensionConstant;
import cn.zzq0324.radish.extension.ExtensionLoader;
import cn.zzq0324.radish.web.annotation.SkipInterceptor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.ResolvableType;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 公众号控制器
 *
 * @author: zzq0324
 * @since : 1.0.0
 */
@Slf4j
@SkipInterceptor
@RestController
@RequestMapping("/api/wechat/callback")
@ConditionalOnProperty(value = "radish.wechat.enable", havingValue = "true")
public class WechatCallbackController {

  @Autowired
  private WechatAppProperties appProperties;

  /**
   * 回调设置的时候会通过GET发起verify url请求，需要原样返回echostr
   */
  @GetMapping("/{appId}")
  public String verifyUrl(@PathVariable("appId") String appId, VerifyUrlRequest request) {
    log.info("appId: {} receive verify url request, request data: {}", appId, JsonUtils.toJson(request));

    // 校验签名
    MessageCrypto crypto = getMessageCrypto(appId);
    crypto.validateSign(request.getTimestamp(), request.getNonce(), request.getSignature());

    return request.getEchostr();
  }

  /**
   * 具体事件回调会通过post回调，整个过程增加事务
   */
  @Transactional
  @PostMapping("/{appId}")
  public Object eventCallback(@PathVariable("appId") String appId, @RequestBody String postData,
      EventCallbackRequest request) {
    MessageCrypto crypto = getMessageCrypto(appId);

    // 校验签名
    crypto.validateSign(request.getTimestamp(), request.getNonce(), request.getSignature());

    // 加密方式，获取xml中的Encrypt字段进行解密
    if (StringUtils.hasLength(request.getEncrypt_type())) {
      postData = XmlUtils.fromXML(postData, AesEncryptData.class).getEncrypt();
    }

    // 得到解密数据
    String data = crypto.decrypt(request.getTimestamp(), request.getNonce(), request.getMsg_signature(), postData);

    log.info("appId: {} receive callback from wechat, callback data: {}", appId, data);

    // 交给扩展点进行处理
    String replyMsg = handleCallback(appId, data);
    if (StringUtils.hasLength(replyMsg)) {
      replyMsg = crypto.encrypt(replyMsg, request.getTimestamp(), request.getNonce());
    }

    return replyMsg;
  }

  private String handleCallback(String appId, String data) {
    // 执行radish自身的逻辑
    BusinessScenario radishScenario = BusinessScenario.of(WechatCallbackHandler.BUSINESS_RADISH);
    executeEventHandler(appId, radishScenario, true, data);

    // 根据回调类型查找对应的扩展实现
    DecryptedData decryptedData = XmlUtils.fromXML(data, DecryptedData.class);

    BusinessScenario bizScenario;
    if (decryptedData.getMsgType().equals(MsgType.EVENT)) {
      bizScenario = BusinessScenario.of(decryptedData.getMsgType(), decryptedData.getEvent(), appId);
    } else {
      bizScenario = BusinessScenario.of(decryptedData.getMsgType(), ExtensionConstant.DEFAULT_USE_CASE, appId);
    }

    return executeEventHandler(appId, bizScenario, true, data);
  }

  private String executeEventHandler(String appId, BusinessScenario scenario, boolean failOver, String data) {
    WechatCallbackHandler handler = ExtensionLoader.getExtension(WechatCallbackHandler.class, scenario, failOver);
    if (handler == null) {
      return "";
    }

    // 根据泛型的具体对象进行转发处理
    ResolvableType resolvableType = ResolvableType.forClass(handler.getClass()).as(WechatCallbackHandler.class);
    Object result = handler.handle(appId, XmlUtils.fromXML(data, resolvableType.getGeneric(0).resolve()));
    if (result == null) {
      return "";
    }

    return XmlUtils.toXML(result);
  }

  private MessageCrypto getMessageCrypto(String appId) {
    AppInfo appInfo = appProperties.getApps().get(appId);

    return new MessageCrypto(appInfo.getToken(), appInfo.getEncodingAesKey(), appId);
  }
}
