/*
 * Copyright (c) 2018-2019 ActionTech.
 * based on code by ServiceComb Pack CopyrightHolder Copyright (C) 2018,
 * License: http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0 or higher.
 */

package org.apache.servicecomb.saga.omega.jdbc.sqlinterceptor.wrapper;

import org.apache.servicecomb.saga.omega.jdbc.sqlinterceptor.info.ResultSetInformation;
import org.apache.servicecomb.saga.omega.jdbc.sqlinterceptor.listener.JdbcEventListener;

import java.sql.*;

/**
 * To wrap the Statement.
 *
 * @author Gannalyo
 * @since 20190129
 */
public class StatementWrapper implements Statement {

    private final Statement statement;
    private final JdbcEventListener eventListener;

    public static Statement wrap(Statement statement, JdbcEventListener eventListener) {
        if (statement == null) {
            return null;
        }
        return new StatementWrapper(statement, eventListener);
    }

    protected StatementWrapper(Statement statement, JdbcEventListener eventListener) {
        this.statement = statement;
        this.eventListener = eventListener;
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            return ResultSetWrapper.wrap(statement.getResultSet(), new ResultSetInformation(), eventListener);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterGetResultSet(System.nanoTime() - start, e);
        }
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeExecuteQuery(sql);
            return ResultSetWrapper.wrap(statement.executeQuery(sql), new ResultSetInformation(), eventListener);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteQuery(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public int[] executeBatch() throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        int[] updateCounts = null;
        try {
            eventListener.onBeforeExecuteBatch();
            updateCounts = statement.executeBatch();
            return updateCounts;
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteBatch(System.nanoTime() - start, updateCounts, e);
        }
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeExecute(sql);
            return statement.execute(sql);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecute(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeExecute(sql);
            return statement.execute(sql, autoGeneratedKeys);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecute(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeExecute(sql);
            return statement.execute(sql, columnIndexes);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecute(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeExecute(sql);
            return statement.execute(sql, columnNames);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecute(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        int rowCount = 0;
        try {
            eventListener.onBeforeExecuteUpdate(sql);
            rowCount = statement.executeUpdate(sql);
            return rowCount;
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteUpdate(System.nanoTime() - start, sql, rowCount, e);
        }
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        int rowCount = 0;
        try {
            eventListener.onBeforeExecuteUpdate(sql);
            rowCount = statement.executeUpdate(sql, autoGeneratedKeys);
            return rowCount;
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteUpdate(System.nanoTime() - start, sql, rowCount, e);
        }
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        int rowCount = 0;
        try {
            eventListener.onBeforeExecuteUpdate(sql);
            rowCount = statement.executeUpdate(sql, columnIndexes);
            return rowCount;
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteUpdate(System.nanoTime() - start, sql, rowCount, e);
        }
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        int rowCount = 0;
        try {
            eventListener.onBeforeExecuteUpdate(sql);
            rowCount = statement.executeUpdate(sql, columnNames);
            return rowCount;
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterExecuteUpdate(System.nanoTime() - start, sql, rowCount, e);
        }
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        SQLException e = null;
        long start = System.nanoTime();
        try {
            eventListener.onBeforeAddBatch(sql);
            statement.addBatch(sql);
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterAddBatch(System.nanoTime() - start, sql, e);
        }
    }

    @Override
    public void close() throws SQLException {
        SQLException e = null;
        try {
            statement.close();
        } catch (SQLException sqlException) {
            e = sqlException;
            throw e;
        } finally {
            eventListener.onAfterStatementClose(e);
        }
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        return statement.getMaxFieldSize();
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        statement.setMaxFieldSize(max);
    }

    @Override
    public int getMaxRows() throws SQLException {
        return statement.getMaxRows();
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        statement.setMaxRows(max);
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        statement.setEscapeProcessing(enable);
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        return statement.getQueryTimeout();
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        statement.setQueryTimeout(seconds);
    }

    @Override
    public void cancel() throws SQLException {
        statement.cancel();
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        return statement.getWarnings();
    }

    @Override
    public void clearWarnings() throws SQLException {
        statement.clearWarnings();
    }

    @Override
    public void setCursorName(String name) throws SQLException {
        statement.setCursorName(name);
    }

    @Override
    public int getUpdateCount() throws SQLException {
        return statement.getUpdateCount();
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        return statement.getMoreResults();
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        statement.setFetchDirection(direction);
    }

    @Override
    public int getFetchDirection() throws SQLException {
        return statement.getFetchDirection();
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        statement.setFetchSize(rows);
    }

    @Override
    public int getFetchSize() throws SQLException {
        return statement.getFetchSize();
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        return statement.getResultSetConcurrency();
    }

    @Override
    public int getResultSetType() throws SQLException {
        return statement.getResultSetType();
    }

    @Override
    public void clearBatch() throws SQLException {
        statement.clearBatch();
    }

    @Override
    public Connection getConnection() throws SQLException {
        return statement.getConnection();
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        return statement.getMoreResults(current);
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        return statement.getGeneratedKeys();
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        return statement.getResultSetHoldability();
    }

    @Override
    public boolean isClosed() throws SQLException {
        return statement.isClosed();
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        statement.setPoolable(poolable);
    }

    @Override
    public boolean isPoolable() throws SQLException {
        return statement.isPoolable();
    }

    @Override
    public void closeOnCompletion() throws SQLException {
        statement.closeOnCompletion();
    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        return statement.isCloseOnCompletion();
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return null;
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
