/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.afrunt.jpa.powerdao;

import javax.persistence.NoResultException;
import javax.persistence.Parameter;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import java.util.*;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * @author Andrii Frunt
 */
public abstract class AbstractExtendedQueryApiDao extends AbstractEntityManagerAdapter implements ExtendedQueryApiDao {

    @Override
    public <ET> Optional<ET> queryResult(String jpql) {
        return queryResult(createQuery(jpql));
    }

    @Override
    public <ET> Optional<ET> queryResult(String jpql, Map<String, Object> params) {
        return queryResult(createQuery(jpql), params);
    }

    @Override
    public <ET> Optional<ET> queryResult(String jpql, List<Object> params) {
        return queryResult(createQuery(jpql), params);
    }

    @Override
    public <ET> Optional<ET> queryResult(String jpql, Class<ET> entityType) {
        return queryResult(createQuery(jpql, entityType));
    }

    @Override
    public <ET> Optional<ET> queryResult(String jpql, Class<ET> entityType, Map<String, Object> params) {
        return queryResult(createQuery(jpql, entityType), params);
    }

    @Override
    public <ET> Optional<ET> queryResult(String jpql, Class<ET> entityType, List<Object> params) {
        return queryResult(createQuery(jpql, entityType), params);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> Optional<ET> queryResult(Query query) {
        try {
            return Optional.ofNullable((ET) query.getSingleResult());
        } catch (NoResultException e) {
            return Optional.empty();
        }
    }

    @Override
    public <ET> Optional<ET> queryResult(Query query, Map<String, Object> params) {
        return queryResult(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Optional<ET> queryResult(Query query, List<Object> params) {
        return queryResult(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Optional<ET> queryResult(TypedQuery<ET> query, Map<String, Object> params) {
        return queryResult(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Optional<ET> queryResult(TypedQuery<ET> query, List<Object> params) {
        return queryResult(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Optional<ET> namedQueryResult(String queryName, Map<String, Object> params) {
        return queryResult(createNamedQuery(queryName), params);
    }

    @Override
    public <ET> Optional<ET> namedQueryResult(String queryName, List<Object> params) {
        return queryResult(createNamedQuery(queryName), params);
    }

    @Override
    public <ET> Optional<ET> namedQueryResult(String queryName, Class<ET> entityType, Map<String, Object> params) {
        return queryResult(createNamedQuery(queryName, entityType), params);
    }

    @Override
    public <ET> Optional<ET> namedQueryResult(String queryName, Class<ET> entityType, List<Object> params) {
        return queryResult(createNamedQuery(queryName, entityType), params);
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql) {
        return queryResult(createNativeQuery(sql));
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql, Class<ET> entityType) {
        return queryResult(createNativeQuery(sql, entityType));
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql, List<Object> params) {
        return queryResult(parameterizedQuery(createNativeQuery(sql), params));
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql, Class<ET> entityType, List<Object> params) {
        return queryResult(parameterizedQuery(createNativeQuery(sql, entityType), params));
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql, String resultSetMapping) {
        return queryResult(createNativeQuery(sql, resultSetMapping));
    }

    @Override
    public <ET> Optional<ET> nativeQueryResult(String sql, String resultSetMapping, List<Object> params) {
        return queryResult(parameterizedQuery(createNativeQuery(sql, resultSetMapping), params));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> List<ET> queryResultList(String jpql) {
        return (List<ET>) queryResultList(createQuery(jpql));
    }

    @Override
    public <ET> List<ET> queryResultList(String jpql, Map<String, Object> params) {
        return queryResultList(createQuery(jpql), params);
    }

    @Override
    public <ET> List<ET> queryResultList(String jpql, List<Object> params) {
        return queryResultList(createQuery(jpql), params);
    }

    @Override
    public <ET> List<ET> queryResultList(String jpql, Class<ET> entityType) {
        return queryResultList(createQuery(jpql, entityType));
    }

    @Override
    public <ET> List<ET> queryResultList(String jpql, Class<ET> entityType, Map<String, Object> params) {
        return queryResultList(parameterizedQuery(createQuery(jpql, entityType), params));
    }

    @Override
    public <ET> List<ET> queryResultList(String jpql, Class<ET> entityType, List<Object> params) {
        return queryResultList(parameterizedQuery(createQuery(jpql, entityType), params));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> List<ET> queryResultList(Query query) {
        return query.getResultList();
    }

    @Override
    public <ET> List<ET> queryResultList(TypedQuery<ET> query) {
        return query.getResultList();
    }

    @Override
    public <ET> List<ET> queryResultList(Query query, Map<String, Object> params) {
        return queryResultList(parameterizedQuery(query, params));
    }

    @Override
    public <ET> List<ET> queryResultList(Query query, List<Object> params) {
        return queryResultList(parameterizedQuery(query, params));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> List<ET> nativeQueryResultList(String sql) {
        return (List<ET>) queryResultList(createNativeQuery(sql));
    }

    @Override
    public <ET> List<ET> nativeQueryResultList(String sql, String resultSetMapping) {
        return queryResultList(createNativeQuery(sql, resultSetMapping));
    }

    @Override
    public <ET> List<ET> nativeQueryResultList(String sql, String resultSetMapping, List<Object> params) {
        return queryResultList(parameterizedQuery(createNativeQuery(sql, resultSetMapping), params));
    }

    @Override
    public <ET> List<ET> nativeQueryResultList(String sql, Class<ET> entityType) {
        return queryResultList(createNativeQuery(sql, entityType));
    }

    @Override
    public <ET> List<ET> nativeQueryResultList(String sql, List<Object> params) {
        return queryResultList(createNativeQuery(sql), params);
    }

    @Override
    public <ET> List<ET> nativeQueryResultList(String sql, Class<ET> entityType, List<Object> params) {
        return queryResultList(createNativeQuery(sql, entityType), params);
    }

    @Override
    public <ET> List<ET> namedQueryResultList(String queryName, Map<String, Object> params) {
        return queryResultList(createNamedQuery(queryName), params);
    }

    @Override
    public <ET> List<ET> namedQueryResultList(String queryName, List<Object> params) {
        return queryResultList(createNamedQuery(queryName), params);
    }


    @Override
    public <ET> List<ET> namedQueryResultList(String queryName) {
        return queryResultList(createNamedQuery(queryName));
    }

    @Override
    public <ET> List<ET> namedQueryResultList(String queryName, Class<ET> entityType) {
        return queryResultList(createNamedQuery(queryName, entityType));
    }

    @Override
    public <ET> List<ET> namedQueryResultList(String queryName, Class<ET> entityType, Map<String, Object> params) {
        return queryResultList(createNamedQuery(queryName, entityType), params);
    }

    @Override
    public <ET> List<ET> namedQueryResultList(String queryName, Class<ET> entityType, List<Object> params) {
        return queryResultList(createNamedQuery(queryName, entityType), params);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql) {
        return queryResultStream(createQuery(jpql));
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql, Map<String, Object> params) {
        return queryResultStream(createQuery(jpql), params);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql, List<Object> params) {
        return queryResultStream(createQuery(jpql), params);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> Stream<ET> queryResultStream(Query query) {
        //TODO: it should be updated after moving to JPA 2.2 Specification
        return query.getResultList().stream();
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET> Stream<ET> queryResultStream(Query query, Map<String, Object> params) {
        return queryResultStream(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Stream<ET> queryResultStream(Query query, List<Object> params) {
        return queryResultStream(parameterizedQuery(query, params));
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql, Class<ET> entityType) {
        return queryResultStream(createQuery(jpql, entityType));
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql, Class<ET> entityType, Map<String, Object> params) {
        return queryResultStream(createQuery(jpql, entityType), params);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(String jpql, Class<ET> entityType, List<Object> params) {
        return queryResultStream(createQuery(jpql, entityType), params);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(Query query, Class<ET> entityType) {
        return queryResultStream(query);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(Query query, Class<ET> entityType, Map<String, Object> params) {
        return queryResultStream(query, params);
    }

    @Override
    public <ET> Stream<ET> queryResultStream(Query query, Class<ET> entityType, List<Object> params) {
        return queryResultStream(query, params);
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name) {
        return queryResultStream(createNamedQuery(name));
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name, Map<String, Object> params) {
        return queryResultStream(createNamedQuery(name), params);
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name, List<Object> params) {
        return queryResultStream(createNamedQuery(name), params);
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name, Class<ET> entityType) {
        return queryResultStream(createNamedQuery(name), entityType);
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name, Class<ET> entityType, Map<String, Object> params) {
        return queryResultStream(createNamedQuery(name), entityType, params);
    }

    @Override
    public <ET> Stream<ET> namedQueryResultStream(String name, Class<ET> entityType, List<Object> params) {
        return queryResultStream(createNamedQuery(name), entityType, params);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET, KT> List<ET> partitionsToNamedQueryResultList(String name, Class<ET> entityType, Collection<KT> params, int partitionSize) {
        return partitionsTo(params, partitionSize, p -> namedQueryResultList(name, entityType, Collections.singletonList(p)));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET, KT> List<ET> partitionsToNamedQueryResultList(String name, Collection<KT> params, int partitionSize) {
        return partitionsTo(params, partitionSize, p -> namedQueryResultList(name, Collections.singletonList(p)));
    }


    @Override
    @SuppressWarnings("unchecked")
    public <ET, KT> List<ET> partitionsToQueryResultList(Query query, Class<ET> entityType, Collection<KT> params, int partitionSize) {
        return partitionsTo(params, partitionSize, p -> queryResultList(passSingleParameter(query, p)));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET, KT> List<ET> partitionsToQueryResultList(String jpql, Class<ET> entityType, Collection<KT> params, int partitionSize) {
        return partitionsToQueryResultList(createQuery(jpql), entityType, params, partitionSize);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <ET, KT> List<ET> partitionsToQueryResultList(String jpql, Collection<KT> params, int partitionSize) {
        Query query = createQuery(jpql);
        return partitionsTo(params, partitionSize, p -> queryResultList(passSingleParameter(query, p)));
    }

    @Override
    public Query parameterizedQuery(Query query, Map<String, Object> params) {
        params.forEach(query::setParameter);
        return query;
    }

    @Override
    public Query parameterizedQuery(Query query, List<Object> params) {
        IntStream.range(0, params.size())
                .boxed()
                .forEach(i -> query.setParameter(i + 1, params.get(i)));

        return query;
    }

    @Override
    public Query passSingleParameter(Query query, Object parameter) {
        Set<Parameter<?>> parameters = query.getParameters();

        if (parameters.size() != 1) {
            throw new IllegalArgumentException("Query should have only one parameter. Actual count is " + parameters.size());
        }

        if (parameters.iterator().next().getName() != null) {
            query.setParameter(parameters.iterator().next().getName(), parameter);
        } else {
            query.setParameter(1, parameter);

        }

        return query;
    }

    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, String jpql) {
        return queryPage(pageIndex, perPage, createQuery(jpql));
    }

    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, String jpql, Map<String, Object> params) {
        return queryPage(pageIndex, perPage, createQuery(jpql), params);
    }

    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, String jpql, List<Object> params) {
        return queryPage(pageIndex, perPage, createQuery(jpql), params);
    }


    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, Query query) {
        int firstResult = (int) (pageIndex * perPage);
        query
                .setFirstResult(firstResult)
                .setMaxResults((int) perPage);

        return queryResultList(query);
    }


    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, Query query, Map<String, Object> params) {
        return queryPage(pageIndex, perPage, parameterizedQuery(query, params));
    }

    @Override
    public <ET> List<ET> queryPage(long pageIndex, long perPage, Query query, List<Object> params) {
        return queryPage(pageIndex, perPage, parameterizedQuery(query, params));
    }


    @Override
    public <ET> List<ET> namedQueryPage(long pageIndex, long perPage, String queryName) {
        return queryPage(pageIndex, perPage, createNamedQuery(queryName));
    }

    @Override
    public <ET> List<ET> namedQueryPage(long pageIndex, long perPage, String queryName, Map<String, Object> params) {
        return queryPage(pageIndex, perPage, parameterizedQuery(createNamedQuery(queryName), params));
    }

    @Override
    public <ET> List<ET> namedQueryPage(long pageIndex, long perPage, String queryName, List<Object> params) {
        return queryPage(pageIndex, perPage, parameterizedQuery(createNamedQuery(queryName), params));
    }
}
