/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.afrunt.jpa.powerdao;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;

/**
 * @author Andrii Frunt
 */
public interface WithCollectionOperations {
    default <KT> List<List<KT>> partitions(Collection<KT> sourceIds, int partitionSize) {
        if (sourceIds == null || sourceIds.isEmpty()) {
            return Collections.emptyList();
        }

        List<KT> ids = new ArrayList<>(sourceIds);

        int idCount = ids.size();

        if (idCount <= partitionSize) {
            return Collections.singletonList(ids);
        }

        int numberOfPartitions = idCount / partitionSize + (idCount % partitionSize == 0 ? 0 : 1);

        return IntStream.range(0, numberOfPartitions)
                .boxed()
                .map(i -> ids.subList(i * partitionSize, i * partitionSize + partitionSize >= idCount ? idCount : i * partitionSize + partitionSize))
                .collect(Collectors.toList());
    }

    default <KT> Stream<List<KT>> partitionsStream(Collection<KT> sourceIds, int partitionSize) {
        return partitions(sourceIds, partitionSize).stream();
    }

    default <KT, ET> List<ET> partitionsTo(Collection<KT> sourceIds, int partitionSize, Function<List<KT>, List<ET>> mapper) {
        return mapPartitions(partitionsStream(sourceIds, partitionSize), mapper);
    }

    default <KT, ET> List<ET> mapPartitions(List<List<KT>> partitions, Function<List<KT>, List<ET>> mapper) {
        return mapPartitions(partitions.stream(), mapper);
    }

    default <KT, ET> List<ET> mapPartitions(Stream<List<KT>> partitions, Function<List<KT>, List<ET>> mapper) {
        return partitions.map(mapper).flatMap(List::stream).collect(Collectors.toList());
    }

    @SuppressWarnings("unchecked")
    default <KT> List<KT> listOf(KT... objs) {
        return Arrays.asList(objs);
    }

    @SuppressWarnings("unchecked")
    default <K, V> Map<K, V> mapOf(Object... kvs) {
        if (kvs == null || kvs.length == 0) {
            return new HashMap<>();
        }

        if (kvs.length % 2 != 0) {
            throw new IllegalArgumentException("Each key should have the value");
        }

        Map<K, V> result = new HashMap<>();

        for (int i = 0; i < kvs.length; i += 2) {
            result.put((K) kvs[i], (V) kvs[i + 1]);
        }


        return result;

    }

    default <K, V> Map<K, V> emptyMap() {
        return mapOf();
    }

    default List<Integer> range(int from, int to) {
        return IntStream.range(from, to).boxed().collect(Collectors.toList());
    }

    default List<Long> range(long from, long to) {
        return LongStream.range(from, to).boxed().collect(Collectors.toList());
    }

    default List<Object> rawRange(int from, int to) {
        return new ArrayList<>(range(from, to));
    }

    default List<Object> rawRange(long from, long to) {
        return new ArrayList<>(range(from, to));
    }
}
