/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.annotations;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import com.anaptecs.jeaf.fastlane.api.FastLane;

@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
@Documented
public @interface FastLaneConfig {
  /**
   * Name of the resource that contains the name of the class with the @ToolsConfig annotation.
   */
  String FAST_LANE_CONFIG_RESOURCE_NAME = "FastLaneConfig";

  /**
   * Path under which the configuration file is stored.
   */
  String FAST_LANE_CONFIG_PATH = FastLane.FASTLANE_BASE_PATH + '/' + FAST_LANE_CONFIG_RESOURCE_NAME;

  /**
   * Default port that is used if nothing else is configured.
   */
  static final int DEFAULT_PORT = 8090;

  /**
   * Default thread idle timeout.
   */
  final int DEFAULT_THREAD_IDLE_TIMEOUT = 5 * 60 * 1000;

  /**
   * Default value for minimum threads of workload container.
   */
  final int DEFAULT_WORKLOAD_MIN_THREADS = 10;

  /**
   * Default value for maximum threads of workload container.
   */
  final int DEFAULT_WORKLOAD_MAX_THREADS = 50;

  /**
   * Default port that is used for the management interface.
   */
  final int DEFAULT_MANAGEMENT_PORT = 8091;

  /**
   * Default value for minimum threads of management container.
   */
  final int DEFAULT_MIN_MANAGEMENT_THREADS = 1;

  /**
   * Default value for maximum threads of management container.
   */
  final int DEFAULT_MAX_MANAGEMENT_THREADS = 8;

  /**
   * Default value for initial queue size.
   */
  final int DEFAULT_INITIAL_QUEUE_SIZE = 100;

  /**
   * Maximum size of queue that is used to store requested until they are really processed. 0 means that the initial
   * size is also the maximum. -1 means that queue is unlimited. It strong recommended to not use this as it might crash
   * the whole server in case of high loads or DoS attacks.
   */
  final int DEFAULT_MAX_QUEUE_SIZE = 0;

  /**
   * Default value for grow size of queue. Size by which the queue is increased when it get's to small.
   */
  final int DEFAULT_QUEUE_GROW_SIZE = 100;

  /**
   * Default size of input buffer when JEAF Fast Lane reads data from incoming requests
   */
  final int DEFAULT_INPUT_BUFFER_SIZE = 8 * 1024;

  /**
   * Default size of output buffer when JEAF Fast Lane writes data as response.
   */
  final int DEFAULT_OUTPUT_BUFFER_SIZE = 32 * 1024;

  /**
   * Default value for context path.
   */
  final String DEFAULT_CONTEXT_PATH = "/*";

  /**
   * Default value for REST root path.
   */
  final String DEFAULT_REST_PATH = "/rest/*";

  /**
   * By default REST services are enabled.
   */
  final boolean DEFAULT_REST_ENABLED = true;

  /**
   * By default Web Servlets are enabled.
   */
  final boolean DEFAULT_WEB_SERVLET_ENABLED = true;

  /**
   * Default path for management interface.
   */
  final String DEFAULT_MANAGEMENT_PATH = "/*";

  /**
   * By default management interface is enabled.
   */
  final boolean DEFAULT_MANAGAMENT_INTERFACE_ENABLED = true;

  /**
   * Default value for HTTP I/O timeout of the HTTP connector. This timeout comes into play if a client does not consume
   * further response data for a certain amount of time.
   */
  final int DEFAULT_HTTP_IO_IDLE_TIMEOUT = 30000;

  /**
   * By default information about server version is not send in response header for security reasons.
   */
  final boolean DEFAULT_SEND_SERVER_VERSION = false;

  /**
   * By default information about server version is not send in response header for security reasons.
   */
  final boolean DEFAULT_SEND_X_POWERED_BY = false;

  /**
   * By default a graceful shutdown is enabled. This allows to process and complete already processed request but new
   * request will not be accepted.
   */
  final boolean DEFAULT_GRACEFUL_SHUTDOWN = true;

  /**
   * Default timeout for graceful shutdown. This timeout defines the time that is waited at a maximum until already
   * processed request will be finished.
   */
  final int DEFAULT_GRACEFUL_SHUTDOWN_TIMEOUT = 30000;

  /**
   * By default JEAF Fast Lane server will be stopped in case that a java.lang.Error occurs.
   */
  final boolean DEFAULT_STOP_SERVER_ON_ERROR = true;

  /**
   * By default java.lang.Error will be traced.
   */
  final boolean DEFAULT_TRACE_UNCAUGHT_ERRORS = true;

  /**
   * By default an uncaught runtime exception will not cause the server to stop.
   */
  final boolean DEFAULT_STOP_SERVER_ON_RUNTIME_EXCEPTION = false;

  /**
   * By default uncaught runtime exceptions will be traced.
   */
  final boolean DEFAULT_TRACE_UNCAUGHT__RUNTIME_EXCEPTIONS = true;

  /**
   * By default JMX is disabled for security reasons.
   */
  final boolean DEFAULT_JMX_ENABLED = false;

  // Settings for workload container

  /**
   * Port that is used to server requests by the so called workload container. The workload container is used to server
   * all kinds of non-management requests.
   */
  int port() default DEFAULT_PORT;

  /**
   * Minimum amount of threads that are used by the workload container.
   */
  int minThreads() default DEFAULT_WORKLOAD_MIN_THREADS;

  /**
   * Maximum amount of threads that are used by the workload container.
   */
  int maxThreads() default DEFAULT_WORKLOAD_MAX_THREADS;

  /**
   * Enables the REST interface so that REST service can be used.
   */
  boolean enableREST() default DEFAULT_REST_ENABLED;

  /**
   * Enables the servlet interface so that Web Servlets can be used.
   */
  boolean enableWebServlets() default DEFAULT_WEB_SERVLET_ENABLED;

  // Settings for management interface

  /**
   * Enables management interface.
   */
  boolean enableManagementInterface() default DEFAULT_MANAGAMENT_INTERFACE_ENABLED;

  /**
   * Port under which the management interface is accessible.
   */
  int managementPort() default DEFAULT_MANAGEMENT_PORT;

  /**
   * Minimum amount of threads that are used for requests to the management interface.
   */
  int minManagementThreads() default DEFAULT_MIN_MANAGEMENT_THREADS;

  /**
   * Maximum amount of threads that are used for requests to the management interface.
   */
  int maxManagementThreads() default DEFAULT_MAX_MANAGEMENT_THREADS;

  /**
   * Idle time after which a thread will be stopped and removed from pool. Timeout is defined in milliseconds.
   */
  int threadIdleTimeout() default DEFAULT_THREAD_IDLE_TIMEOUT;

  /**
   * Initial size of queue that is used to store requests until they are really processed.
   */
  int initialQueueSize() default DEFAULT_INITIAL_QUEUE_SIZE;

  /**
   * Maximum size of queue that is used to store requested until they are really processed.
   * 
   * <ul>
   * <li>0 means that the initial size is also the maximum.</li>
   * <li>-1 means that queue is unlimited. It strong recommended to not use this as it might crash the whole server in
   * case of high loads or DoS attacks.</li>
   * </ul>
   * 
   */
  int maxQueueSize() default DEFAULT_MAX_QUEUE_SIZE;

  /**
   * Size of which the request queue should be increased until the maximum is reached.
   */
  int queueGrowSize() default DEFAULT_QUEUE_GROW_SIZE;

  /**
   * Size of input buffer that is used when request data is read.
   */
  int inputBufferSize() default DEFAULT_INPUT_BUFFER_SIZE;

  /**
   * Size of output buffer that is used when response data is written.
   */
  int outputBufferSize() default DEFAULT_OUTPUT_BUFFER_SIZE;

  /**
   * Context path of the workload container.
   */
  String contextPath() default DEFAULT_CONTEXT_PATH;

  /**
   * Root path under which REST resources will be available.
   */
  String restPath() default DEFAULT_REST_PATH;

  /**
   * Context path under which the management interface will be available.
   */
  String managementPath() default DEFAULT_MANAGEMENT_PATH;

  /**
   * HTTP I/O timeout of the HTTP connector. This timeout comes into play if a client does not consume further response
   * data for a certain amount of time.
   */
  int httpIOIdleTimeout() default DEFAULT_HTTP_IO_IDLE_TIMEOUT;

  /**
   * Enables sending the server version as a response header field. It's recommended to not enable this info for
   * security reasons.
   */
  boolean sendServerVersion() default DEFAULT_SEND_SERVER_VERSION;

  /**
   * Enables sending the server version as a response header field. It's recommended to not enable this info for
   * security reasons.
   */
  boolean sendXPoweredBy() default DEFAULT_SEND_X_POWERED_BY;

  /**
   * Enables graceful shutdown behavior. Graceful shutdown means that request that are already processed will be
   * finished but new ones will not be accepted.
   */
  boolean gracefulShutdown() default DEFAULT_GRACEFUL_SHUTDOWN;

  /**
   * Timeout for graceful shutdown. If this timeout is exceeded then running threads will be killed.
   */
  int gracefulShutdownTimeout() default DEFAULT_GRACEFUL_SHUTDOWN_TIMEOUT;

  /**
   * Defines if JEAF Fast Lane server should be stopped when a {@link java.lang.Error} occurs.
   */
  boolean stopServerOnError() default DEFAULT_STOP_SERVER_ON_ERROR;

  /**
   * Defines if uncaught {@link java.lang.Error} should be traced.
   */
  boolean traceUncaughtErrors() default DEFAULT_TRACE_UNCAUGHT_ERRORS;

  /**
   * Defines if JEAF Fast Lane server should be stopped when a {@link java.lang.RuntimeException} occurs.
   */
  boolean stopServerOnRuntimeException() default DEFAULT_STOP_SERVER_ON_RUNTIME_EXCEPTION;

  /**
   * Defines if uncaught {@link java.lang.RuntimeException} should be traced.
   */
  boolean traceUncaughtRuntimeExceptions() default DEFAULT_TRACE_UNCAUGHT__RUNTIME_EXCEPTIONS;

  /**
   * Defines if JMX should be enabled or not. By default JMX is disabled to prevent security issues.
   */
  boolean enableJMX() default DEFAULT_JMX_ENABLED;

  /**
   * Optional additional URL mappings for servlets. Servlet mappings are an addition to what is already configured in
   * the @WebServlet annotation.
   */
  ServletMapping[] servletMappings() default {};
}
