/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.util.Arrays;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.anaptecs.jeaf.xfun.api.XFun;
import com.anaptecs.jeaf.xfun.api.checks.Assert;
import com.anaptecs.jeaf.xfun.api.info.ApplicationInfo;
import com.anaptecs.jeaf.xfun.api.info.InfoProvider;

/**
 * Class implements a very basic REST based management interface for JEAF FastLane.
 * 
 * @author JEAF Development Team
 */
@Path("management")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class ManagementResource {
  /**
   * Servlet context is used to resolve reference to JEAf Fast Lane Server instance.
   */
  @Context
  private ServletContext context;

  /**
   * Method returns reference to JEAF Fast Lane Server instance that is managed by this resource.
   * 
   * @return {@link FastLaneServer} Reference to JEAF Fast Lane Server instance. The method never returns null.
   */
  private FastLaneServer getFastLaneServer( ) {
    FastLaneServer lFastLaneServer = (FastLaneServer) context.getAttribute(ManagementFilter.FAST_LANE_SERVER_KEY);
    Assert.assertNotNull(lFastLaneServer, "JEAF Fast Lane Server instance must be available through servlet context.");
    return lFastLaneServer;
  }

  /**
   * Method returns general information about the application
   * 
   * @return {@link ApplicationInfo} General information. The method never returns null.
   * 
   * {@link InfoProvider#getApplicationInfo()}
   */
  @GET
  @Path("info")
  public ApplicationInfo getInfo( ) {
    return XFun.getInfoProvider().getApplicationInfo();
  }

  /**
   * Method returns information about the state of the workload container.
   * 
   * @return {@link WebContainerStateInfo} Status information about the workload container. The method never returns null.
   */
  @GET
  @Path("workloadContainer")
  public WebContainerStateInfo getWorkloadContainerState( ) {
    return this.getFastLaneServer().getWorkloadContainer().getState();
  }

  /**
   * Method returns information about the state of the management container.
   * 
   * @return {@link WebContainerStateInfo} Status information about the management container. The method never returns null.
   */
  @GET
  @Path("managementContainer")
  public WebContainerStateInfo getManagementContainerState( ) {
    return this.getFastLaneServer().getManagementContainer().getState();
  }

  /**
   * Method initiates the shutdown of the JEAF Fast Lane server.
   */
  @DELETE
  @Path("shutdown")
  public void shutdown( ) {
    // Request shutdown of server. As this is a regular shutdown exit code should be 0.
    this.getFastLaneServer().requestShutdown(50, 0);
  }

  @GET
  @Path("health")
  public Response getHealthState( ) {
    HealthInfo lLivenessState = this.getFastLaneServer().getWorkloadContainer().getLivenessState();
    HealthInfo lReadinessState = this.getFastLaneServer().getWorkloadContainer().getReadinessState();

    HealthInfoState lOverallState;
    if (lLivenessState.getStatus() == HealthInfoState.UP && lReadinessState.getStatus() == HealthInfoState.UP) {
      lOverallState = HealthInfoState.UP;
    }
    else {
      lOverallState = HealthInfoState.OUT_OF_SERVICE;
    }

    ApplicationInfo lApplicationInfo = XFun.getInfoProvider().getApplicationInfo();
    String lVersionString = lApplicationInfo.getVersion().getVersionString();
    HealthInfoGroup lHealthInfoGroup = new HealthInfoGroup(lApplicationInfo.getName(), lVersionString, lOverallState,
        Arrays.asList(lLivenessState, lReadinessState));

    return Response.status(lOverallState.asHTTPStatus()).entity(lHealthInfoGroup).build();
  }

  @GET
  @Path("health/liveness")
  public Response getLivenessState( ) {
    HealthInfo lLivenessState = this.getFastLaneServer().getWorkloadContainer().getLivenessState();
    return Response.status(lLivenessState.getStatus().asHTTPStatus()).entity(lLivenessState).build();
  }

  @GET
  @Path("health/readiness")
  public Response getReadinessState( ) {
    HealthInfo lReadinessState = this.getFastLaneServer().getWorkloadContainer().getReadinessState();
    return Response.status(lReadinessState.getStatus().asHTTPStatus()).entity(lReadinessState).build();
  }
}
