/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.servlet.Servlet;
import javax.servlet.annotation.WebInitParam;
import javax.servlet.annotation.WebServlet;

import com.anaptecs.jeaf.fastlane.annotations.ServletMapping;
import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.xfun.api.checks.Check;

public class WebServletInfo {
  private final Class<? extends Servlet> servletClass;

  private final String displayName;

  private final boolean asyncSupported;

  private final int loadOnStartup;

  private final Map<String, String> initParams;

  private final SortedSet<String> urlPatterns;

  private WebServletInfo( Builder pBuilder ) {
    // Check parameters
    Check.checkInvalidParameterNull(pBuilder.servletClass, "pBuilder.servletClass");
    Check.checkInvalidParameterNull(pBuilder.initParams, "pBuilder.initParams");
    Check.checkInvalidParameterNull(pBuilder.urlPatterns, "pBuilder.urlPatterns");

    servletClass = pBuilder.servletClass;

    if (Tools.getStringTools().isRealString(pBuilder.displayName) == true) {
      displayName = pBuilder.displayName;
    }
    else {
      displayName = servletClass.getSimpleName();
    }
    asyncSupported = pBuilder.asyncSupported;
    loadOnStartup = pBuilder.loadOnStartup;
    initParams = Collections.unmodifiableMap(new HashMap<>(pBuilder.initParams));
    urlPatterns = Collections.unmodifiableSortedSet(new TreeSet<>(pBuilder.urlPatterns));
  }

  public Class<? extends Servlet> getServletClass( ) {
    return servletClass;
  }

  public String getDisplayName( ) {
    return displayName;
  }

  public boolean isAsyncSupported( ) {
    return asyncSupported;
  }

  public int getLoadOnStartup( ) {
    return loadOnStartup;
  }

  public Map<String, String> getInitParams( ) {
    return initParams;
  }

  public SortedSet<String> getURLPatterns( ) {
    return urlPatterns;
  }

  static class Builder {
    private Class<? extends Servlet> servletClass;

    private String displayName;

    private boolean asyncSupported;

    private int loadOnStartup;

    private final Map<String, String> initParams = new HashMap<>();

    private final Set<String> urlPatterns = new HashSet<>();

    public static Builder newBuilder( ) {
      return new Builder();
    }

    private Builder( ) {
      // Nothing to do.
    }

    public Builder setServletClass( Class<? extends Servlet> pServletClass ) {
      servletClass = pServletClass;
      return this;
    }

    public Builder setFromServletAnnotation( WebServlet pAnnotation ) {
      displayName = pAnnotation.displayName();
      asyncSupported = pAnnotation.asyncSupported();
      loadOnStartup = pAnnotation.loadOnStartup();

      for (WebInitParam lParam : pAnnotation.initParams()) {
        initParams.put(lParam.name(), lParam.value());
      }

      urlPatterns.addAll(Arrays.asList(pAnnotation.value()));
      urlPatterns.addAll(Arrays.asList(pAnnotation.urlPatterns()));
      return this;
    }

    public Builder addServletMappings( ServletMapping pServletMapping ) {
      if (pServletMapping != null) {
        urlPatterns.addAll(Arrays.asList(pServletMapping.urlPatterns()));
      }
      return this;
    }

    public WebServletInfo build( ) {
      return new WebServletInfo(this);
    }
  }

}
