/**
 * Copyright 2004 - 2020 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.lang.management.ManagementFactory;
import java.util.Arrays;
import java.util.List;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.servlet.Filter;
import javax.servlet.Servlet;

import com.anaptecs.jeaf.fastlane.annotations.ServletMapping;
import com.anaptecs.jeaf.fastlane.api.FastLane;
import com.anaptecs.jeaf.fastlane.impl.WebContainer.Builder;
import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.tools.api.performance.Stopwatch;
import com.anaptecs.jeaf.tools.api.performance.TimePrecision;
import com.anaptecs.jeaf.xfun.api.XFun;
import com.anaptecs.jeaf.xfun.api.checks.Assert;
import com.anaptecs.jeaf.xfun.api.checks.Check;
import com.anaptecs.jeaf.xfun.api.config.ConfigurationReader;

public class FastLaneServer implements FastLaneServerMBean {
  /**
   * Constant for name of workload container.
   */
  public static final String WORKLOAD_CONTAINER_NAME = "Workload Container";

  /**
   * Constant for name of management container
   */
  public static final String MANAGEMENT_CONTAINER_NAME = "Management Container";

  /**
   * Constant for the name of the Server MBean.
   */
  public static final String SERVER_MBEAN_NAME = "com.anaptecs.jeaf.fastlane:name=JEAF Fast Lane Server";

  /**
   * Constant for first part of MBean name for web containers.
   */
  public static final String CONTAINER_PREFIX = "com.anaptecs.jeaf.fastlane:type=Containers, name=";

  /**
   * Constant for name of workload container MBean.
   */
  public static final String WORKLOAD_CONTAINER_MBEAN_NAME = CONTAINER_PREFIX + WORKLOAD_CONTAINER_NAME;

  /**
   * Constant for name of workload container MBean.
   */
  public static final String MANAGEMENT_CONTAINER_MBEAN_NAME = CONTAINER_PREFIX + MANAGEMENT_CONTAINER_NAME;

  /**
   * Workload container is used to process incoming requests.
   */
  private WebContainer workloadContainer;

  /**
   * Management container is used to host the management interface. As the management interface is hosted inside an
   * other container it is only accessible via a different port.
   */
  private WebContainer managementContainer;

  /**
   * Current state of the server.
   */
  private WebContainerState state;

  /**
   * Exit code in case that the instance was stopped. By default it is 0.
   */
  private int exitCode = 0;

  /**
   * Attribute defines if JMX is enabled or not.
   */
  private boolean jmxStarted;

  /**
   * Initialize object using default configuration mechanism.
   * 
   * @see FastLaneConfiguration
   */
  public FastLaneServer( ) {
    this(new AnnotationBasedFastLaneConfigurationImpl());
  }

  public FastLaneServer( FastLaneConfiguration pConfiguration ) {
    // Check parameter
    Check.checkInvalidParameterNull(pConfiguration, "pConfiguration");

    // Create management container. In case that management interface is disabled the method will return null.
    managementContainer = this.createManagementContainer(pConfiguration);

    // Create workload container based on configuration
    workloadContainer = this.createWorkloadContainer(pConfiguration);

    state = WebContainerState.STOPPED;

    // Register JMX MBeans
    if (pConfiguration.isJMXEnabled()) {
      this.startJMX();
    }
    else {
      XFun.getTrace().info("Skipping MBeans registration as JMX is disabled.");
    }
  }

  private WebContainer createWorkloadContainer( FastLaneConfiguration pConfiguration ) {
    // Check parameter
    Assert.assertNotNull(pConfiguration, "pConfiguration");

    // Configure workload container.
    Builder lBuilder = Builder.newBuilder();
    lBuilder.setContextPath(pConfiguration.getContextPath());
    lBuilder.setName(WORKLOAD_CONTAINER_NAME);
    lBuilder.setPort(pConfiguration.getPort());
    lBuilder.setMinThreads(pConfiguration.getMinThreads());
    lBuilder.setMaxThreads(pConfiguration.getMaxThreads());
    lBuilder.setThreadIdleTimeout(pConfiguration.getThreadIdleTimeout());
    lBuilder.setThreadPoolName("worker");
    lBuilder.setInitialQueueSize(pConfiguration.getInitialQueueSize());
    lBuilder.setMaxQueueSize(pConfiguration.getMaxQueueSize());
    lBuilder.setQueueGrowSize(pConfiguration.getQueueGrowSize());
    lBuilder.setInputBufferSize(pConfiguration.getInputBufferSize());
    lBuilder.setOutputBufferSize(pConfiguration.getOutputBufferSize());
    lBuilder.setSendServerVersion(pConfiguration.sendServerVersion());
    lBuilder.setSendXPoweredBy(pConfiguration.sendXPoweredBy());
    lBuilder.setGracefulShutdownEnabled(pConfiguration.isGracefulShutdownEnabled());
    lBuilder.setGracefulShutdownTimeout(pConfiguration.getGracefulShutdownTimeout());
    WebContainer lWorkloadContainer = lBuilder.build();

    // Add filter for error handling
    boolean lStopServerOnError = pConfiguration.stopServerOnError();
    boolean lTraceUncaughtErrors = pConfiguration.traceUncaughtErrors();
    boolean lStopServerOnRuntimeException = pConfiguration.stopServerOnRuntimeException();
    boolean lTraceUncaughtRuntimeExceptions = pConfiguration.traceUncaughtRuntimeExceptions();
    ErrorHandlerFilter lErrorHandlerFilter = new ErrorHandlerFilter(this, lStopServerOnError, lTraceUncaughtErrors,
        lStopServerOnRuntimeException, lTraceUncaughtRuntimeExceptions);
    lWorkloadContainer.addFilter(ErrorHandlerFilter.class, lErrorHandlerFilter);

    // Add servlets
    ConfigurationReader lReader = new ConfigurationReader();
    List<Class<? extends Servlet>> lServlets =
        lReader.readClassesFromConfigFile(FastLane.SERVLET_CONFIG_PATH, Servlet.class);
    List<ServletMapping> lServletMappings = pConfiguration.getServletMappings();
    lWorkloadContainer.addServlets(lServlets, lServletMappings);

    // Add REST resources
    List<Class<?>> lRESTResources = lReader.readClassesFromConfigFile(FastLane.REST_RESOURCE_CONFIG_PATH);
    lWorkloadContainer.addRESTResources(pConfiguration.getRESTPath(), lRESTResources);

    // Add filters
    List<Class<? extends Filter>> lFilters =
        lReader.readClassesFromConfigFile(FastLane.FILTER_CONFIG_PATH, Filter.class);
    lWorkloadContainer.addFilters(lFilters);
    return lWorkloadContainer;
  }

  private WebContainer createManagementContainer( FastLaneConfiguration pConfiguration ) {
    // Check parameter
    Assert.assertNotNull(pConfiguration, "pConfiguration");

    // Check if management container should be used.
    WebContainer lManagementContainer;
    if (pConfiguration.isManagementInterfaceEnabled() == true) {
      // Configure workload container.
      Builder lBuilder = Builder.newBuilder();
      lBuilder.setContextPath(pConfiguration.getManagementPath());
      lBuilder.setName(MANAGEMENT_CONTAINER_NAME);
      lBuilder.setPort(pConfiguration.getManagementPort());
      lBuilder.setMinThreads(pConfiguration.getMinManagementThreads());
      lBuilder.setMaxThreads(pConfiguration.getMaxManagementThreads());
      lBuilder.setThreadIdleTimeout(pConfiguration.getThreadIdleTimeout());
      lBuilder.setThreadPoolName("management");
      lBuilder.setInitialQueueSize(pConfiguration.getInitialQueueSize());
      lBuilder.setMaxQueueSize(pConfiguration.getMaxQueueSize());
      lBuilder.setQueueGrowSize(pConfiguration.getQueueGrowSize());
      lBuilder.setInputBufferSize(pConfiguration.getInputBufferSize());
      lBuilder.setOutputBufferSize(pConfiguration.getOutputBufferSize());
      lBuilder.setSendServerVersion(pConfiguration.sendServerVersion());
      lBuilder.setSendXPoweredBy(pConfiguration.sendXPoweredBy());
      lBuilder.setGracefulShutdownEnabled(pConfiguration.isGracefulShutdownEnabled());
      lBuilder.setGracefulShutdownTimeout(pConfiguration.getGracefulShutdownTimeout());
      lManagementContainer = lBuilder.build();

      // Add special filter that ensure that management REST resource have access to the server instance.
      ManagementFilter lManagementFilter = new ManagementFilter(this);
      lManagementContainer.addFilter(ManagementFilter.class, lManagementFilter);

      // Add management REST resource
      lManagementContainer.addRESTResources(pConfiguration.getManagementPath(),
          Arrays.asList(ManagementResource.class));
    }
    else {
      lManagementContainer = null;
    }
    return lManagementContainer;
  }

  public synchronized void start( ) {
    state = WebContainerState.STARTING;
    XFun.getTrace().info("Starting JEAF Fast Lane Server");

    // Start containers
    if (managementContainer != null) {
      managementContainer.start();
    }

    workloadContainer.start();
    state = WebContainerState.RUNNING;
    XFun.getTrace().info("JEAF Fast Lane startup completed.");
  }

  public int join( ) {
    if (managementContainer != null) {
      managementContainer.join();
    }
    else {
      workloadContainer.join();
    }
    return exitCode;
  }

  public int startAndJoin( ) {
    // Start server and wait until it is stopped again.
    int lExitCode;
    try {
      this.start();
      lExitCode = this.join();
    }
    catch (RuntimeException | Error e) {
      XFun.getTrace().error("Exception when trying to start JEAF Fast Lane server.", e);
      lExitCode = -1;
    }
    return lExitCode;
  }

  public void requestShutdown( int pWaitTime, int pExitCode ) {
    // Create new shutdown thread.
    Thread lShutdownThread = new Thread(new Runnable() {

      @Override
      public void run( ) {
        // Wait for caller to complete
        try {
          state = WebContainerState.STOPPING;
          Thread.sleep(pWaitTime);
          shutdown(pExitCode);
        }
        catch (InterruptedException e) {
          Thread.currentThread().interrupt();
        }
      }
    }, "shutdown");

    // Run shutdown thread.
    XFun.getTrace().info("Shutdown requested");
    lShutdownThread.start();
  }

  public synchronized void shutdown( int pExitCode ) {
    Stopwatch lStopwatch =
        Tools.getPerformanceTools().createStopwatch("JEAF FastLane Shutdown took", TimePrecision.MILLIS).start();
    state = WebContainerState.STOPPING;
    exitCode = pExitCode;

    // Stop workload container. If graceful shutdown in enabled this may take a while.
    workloadContainer.stop();

    // Stop management container. If graceful shutdown in enabled this may take a while.
    if (managementContainer != null) {
      managementContainer.stop();
    }

    // Unregister JMX MBeans
    this.stopJMX();

    state = WebContainerState.STOPPED;
    lStopwatch.stopAndTrace();
  }

  /**
   * Method starts JMX MBeans of JEAF Fast Lane server.
   */
  private void startJMX( ) {
    try {
      XFun.getTrace().info("Registering JEAF Fast Lane MBeans.");
      MBeanServer lMBeanServer = ManagementFactory.getPlatformMBeanServer();
      lMBeanServer.registerMBean(this, new ObjectName(SERVER_MBEAN_NAME));

      // Register workload container
      lMBeanServer.registerMBean(workloadContainer, new ObjectName(WORKLOAD_CONTAINER_MBEAN_NAME));

      if (managementContainer != null) {
        lMBeanServer.registerMBean(managementContainer, new ObjectName(MANAGEMENT_CONTAINER_MBEAN_NAME));
      }
    }
    catch (JMException e) {
      XFun.getTrace().error(e.getMessage(), e);
    }

    // Set JMX status.
    jmxStarted = true;
  }

  /**
   * Method stops JMX MBeans
   */
  private void stopJMX( ) {
    if (jmxStarted == true) {
      try {
        MBeanServer lMBeanServer = ManagementFactory.getPlatformMBeanServer();
        lMBeanServer.unregisterMBean(new ObjectName(SERVER_MBEAN_NAME));

        // Register workload container
        lMBeanServer.unregisterMBean(new ObjectName(WORKLOAD_CONTAINER_MBEAN_NAME));

        if (managementContainer != null) {
          lMBeanServer.unregisterMBean(new ObjectName(MANAGEMENT_CONTAINER_MBEAN_NAME));
        }
      }
      catch (JMException e) {
        XFun.getTrace().error(e.getMessage(), e);
      }
      // Set JMX status
      jmxStarted = false;
    }
    // JMX was not started.
    else {
      // Nothing to do.
    }
  }

  public WebContainerState getState( ) {
    return state;
  }

  public WebContainer getWorkloadContainer( ) {
    return workloadContainer;
  }

  public WebContainer getManagementContainer( ) {
    return managementContainer;
  }

  @Override
  public String getServerState( ) {
    return state.getName();
  }

  public static void main( String[] pArgs ) {
    // Start server and wait until is is shutdown again.
    FastLaneServer lServer = new FastLaneServer();

    int lExitCode = lServer.startAndJoin();
    XFun.getTrace().info("Exiting JVM of JEAF Fast Lane Server with exit code " + lExitCode);
    System.exit(lExitCode);
  }
}
