/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;

import javax.servlet.Filter;
import javax.servlet.Servlet;
import javax.servlet.annotation.WebFilter;
import javax.servlet.annotation.WebServlet;

import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.servlet.FilterHolder;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.thread.MonitoredQueuedThreadPool;
import org.glassfish.jersey.servlet.ServletContainer;

import com.anaptecs.jeaf.fastlane.annotations.FastLaneConfig;
import com.anaptecs.jeaf.fastlane.annotations.ServletMapping;
import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.xfun.api.XFun;
import com.anaptecs.jeaf.xfun.api.checks.Assert;
import com.anaptecs.jeaf.xfun.api.checks.Check;
import com.anaptecs.jeaf.xfun.api.errorhandling.JEAFSystemException;

public class WebContainer implements WebContainerMBean {
  private static final List<Class<? extends Filter>> ignoredFilters =
      Arrays.asList(GracefulShutdownFilter.class, ManagementFilter.class, ErrorHandlerFilter.class);

  private final String name;

  private final ServerConnector connector;

  private final MonitoredQueuedThreadPool threadPool;

  private final Server server;

  private final ServletContextHandler contextHandler;

  private final List<Class<?>> restResources = new ArrayList<>();

  private final List<WebServletInfo> servlets = new ArrayList<>();

  private final List<WebFilterInfo> filters = new ArrayList<>();

  private final GracefulShutdownFilter gracefulShutdownFilter;

  private final boolean gracefulShutdownEnabled;

  private final int gracefulShutdownTimeout;

  private final int maxQueueSize;

  public WebContainer( Builder pBuilder ) {
    name = pBuilder.name;

    // Create thread pool and server
    maxQueueSize = this.getMaxQueueCapacity(pBuilder);
    BlockingQueue<Runnable> lQueue =
        new BlockingArrayQueue<>(pBuilder.initialQueueSize, pBuilder.queueGrowSize, maxQueueSize);
    threadPool =
        new MonitoredQueuedThreadPool(pBuilder.maxThreads, pBuilder.minThreads, pBuilder.threadIdleTimeout, lQueue);
    threadPool.setName(pBuilder.threadPoolName);
    server = new Server(threadPool);

    // Create connector
    HttpConfiguration lHttpConfiguration = new HttpConfiguration();
    lHttpConfiguration.setSendServerVersion(pBuilder.sendServerVersion);
    lHttpConfiguration.setSendXPoweredBy(pBuilder.sendXPoweredBy);
    lHttpConfiguration.setIdleTimeout(pBuilder.httpIOIdleTimeout);
    lHttpConfiguration.setOutputBufferSize(pBuilder.outputBufferSize);
    HttpConnectionFactory lFactory = new HttpConnectionFactory(lHttpConfiguration);
    lFactory.setInputBufferSize(pBuilder.inputBufferSize);
    connector = new ServerConnector(server, lFactory);
    connector.setPort(pBuilder.port);
    connector.setName(pBuilder.name);
    connector.setAcceptQueueSize(200);
    server.setConnectors(new Connector[] { connector });

    // Trace information about buffer sizes.
    XFun.getTrace().info("Jetty Input buffer size: " + lFactory.getInputBufferSize());
    XFun.getTrace().info("Jetty Input buffer size: " + lHttpConfiguration.getOutputBufferSize());

    // Create context handler that supports GZIP compression.
    int lOptions = ServletContextHandler.SESSIONS | ServletContextHandler.GZIP;
    contextHandler = new ServletContextHandler(lOptions);
    contextHandler.setContextPath(pBuilder.contextPath);
    server.setHandler(contextHandler);

    // Configure shutdown behavior
    gracefulShutdownEnabled = pBuilder.gracefulShutdownEnabled;
    gracefulShutdownTimeout = pBuilder.gracefulShutdownTimeout;
    server.setStopAtShutdown(gracefulShutdownEnabled);
    server.setStopTimeout(gracefulShutdownTimeout);

    // Configure handler depending on shutdown behavior.
    if (gracefulShutdownEnabled == true) {
      gracefulShutdownFilter = new GracefulShutdownFilter();
      this.addFilter(GracefulShutdownFilter.class, gracefulShutdownFilter);
    }
    else {
      gracefulShutdownFilter = null;
    }
  }

  /**
   * Method creates a new BlockingQueue that will be used to store request until they will be executed.
   * 
   * @param pBuilder Builder is used to resolve configuration of queue. The parameter must not be null.
   * @return
   */
  private int getMaxQueueCapacity( Builder pBuilder ) {
    int lInitialQueueSize = pBuilder.initialQueueSize;
    int lMaxQueueSize = pBuilder.maxQueueSize;

    if (lInitialQueueSize > 0 && lMaxQueueSize == 0) {
      lMaxQueueSize = lInitialQueueSize;
    }
    else if (lInitialQueueSize > lMaxQueueSize) {
      XFun.getTrace().warn(
          "Wrong max queue size defined in configuration. Initial queue size is bigger than maximum. Using initial queue size also as maximum size.");
      lMaxQueueSize = lInitialQueueSize;
    }
    return lMaxQueueSize;
  }

  public String getName( ) {
    return name;
  }

  public List<Class<?>> getRESTResources( ) {
    return restResources;
  }

  public List<WebServletInfo> getServlets( ) {
    return servlets;
  }

  public List<WebFilterInfo> getFilters( ) {
    return filters;
  }

  public List<Class<?>> addRESTResources( String pRESTPath, List<Class<?>> pRESTResourceClasses ) {
    // Enable REST services using Jersey
    ServletHolder lJerseyServlet = contextHandler.addServlet(ServletContainer.class, pRESTPath);
    lJerseyServlet.setInitOrder(1);
    lJerseyServlet.setAsyncSupported(true);

    // Lookup all rest resources
    String lAllClassNames = Tools.getStringTools().getClassNamesAsString(pRESTResourceClasses);
    lJerseyServlet.setInitParameter("jersey.config.server.provider.classnames", lAllClassNames);
    // We don't want Jersey to support WADL
    lJerseyServlet.setInitParameter("jersey.config.server.wadl.disableWadl", "true");

    for (Class<?> lNextClass : pRESTResourceClasses) {
      restResources.add(lNextClass);
      XFun.getTrace().info("Added REST resource " + lNextClass.getName());
    }
    return pRESTResourceClasses;
  }

  public List<WebServletInfo> addServlets( List<Class<? extends Servlet>> pServletClasses,
      List<ServletMapping> pServletMappings ) {

    // Check parameters.
    Check.checkInvalidParameterNull(pServletClasses, "pServletClasses");

    // Build map with servlet mappings
    Map<Class<? extends Servlet>, ServletMapping> lMappings = new HashMap<>();
    if (pServletMappings != null) {
      for (ServletMapping lNextMapping : pServletMappings) {
        lMappings.put(lNextMapping.servletClass(), lNextMapping);
      }
    }

    // Process all servlets
    List<WebServletInfo> lAddedServlets = new ArrayList<>(pServletClasses.size());
    for (Class<? extends Servlet> lNextServletClass : pServletClasses) {
      WebServlet lAnnotation = lNextServletClass.getAnnotation(WebServlet.class);
      if (lAnnotation != null) {
        // Create servlet info.
        com.anaptecs.jeaf.fastlane.impl.WebServletInfo.Builder lBuilder = WebServletInfo.Builder.newBuilder();
        lBuilder.setServletClass(lNextServletClass);
        lBuilder.setFromServletAnnotation(lAnnotation);
        lBuilder.addServletMappings(lMappings.get(lNextServletClass));
        WebServletInfo lServletInfo = lBuilder.build();

        // Register servlet for all provided URLs.
        this.registerServlet(lServletInfo);
        lAddedServlets.add(lServletInfo);
      }
      // Ignoring servlet class as it does not have the expected annotation.
      else {
        XFun.getTrace().error("Ignoring servlet class " + lNextServletClass.getName()
            + " as it is not annotated with required annotation @WebServlet");
      }
    }

    // Return list of added servlets
    return lAddedServlets;
  }

  private void registerServlet( WebServletInfo pWebServletInfo ) {
    for (String lURLPattern : pWebServletInfo.getURLPatterns()) {
      ServletHolder lServlet = contextHandler.addServlet(pWebServletInfo.getServletClass(), lURLPattern);
      lServlet.setAsyncSupported(pWebServletInfo.isAsyncSupported());
      lServlet.setDisplayName(pWebServletInfo.getDisplayName());
      lServlet.setInitOrder(pWebServletInfo.getLoadOnStartup());
      lServlet.setInitParameters(pWebServletInfo.getInitParams());
      XFun.getTrace().info("Added servlet " + pWebServletInfo.getDisplayName() + " under path '" + lURLPattern
          + "' using implementation " + pWebServletInfo.getServletClass().getName());
    }
    servlets.add(pWebServletInfo);

  }

  public List<WebFilterInfo> addFilters( List<Class<? extends Filter>> pFilterClasses ) {
    // Check parameters.
    Check.checkInvalidParameterNull(pFilterClasses, "pFilterClasses");

    List<WebFilterInfo> lAddedFilters = new ArrayList<>(pFilterClasses.size());
    for (Class<? extends Filter> lNextFilterClass : pFilterClasses) {
      // Avoid that filter for graceful shutdown is added by accident.
      if (this.ignoreFilterClass(lNextFilterClass) == false) {
        WebFilterInfo lFilterInfo = this.addFilter(lNextFilterClass, null);
        if (lFilterInfo != null) {
          lAddedFilters.add(lFilterInfo);
        }
      }
    }

    // Return list of added filters.
    return lAddedFilters;
  }

  private boolean ignoreFilterClass( Class<? extends Filter> pFilterClass ) {
    return ignoredFilters.contains(pFilterClass);
  }

  public WebFilterInfo addFilter( Class<? extends Filter> pFilterClass, Filter pFilterInstance ) {
    WebFilter lAnnotation = pFilterClass.getAnnotation(WebFilter.class);
    WebFilterInfo lFilterInfo;
    if (lAnnotation != null) {
      // Resolve all information about the filter that should be created.
      com.anaptecs.jeaf.fastlane.impl.WebFilterInfo.Builder lBuilder = WebFilterInfo.Builder.newBuilder();
      lBuilder.setFilterClass(pFilterClass);
      lBuilder.setFromFilterAnnotation(lAnnotation);
      lFilterInfo = lBuilder.build();

      // Register filter for all provided paths
      this.registerFilter(lFilterInfo, pFilterInstance);
    }
    // Filter does not have required annotation
    else {
      lFilterInfo = null;
      XFun.getTrace().error("Ignoring servlet filter class " + pFilterClass.getName()
          + " as it is not annotated with required annotation @WebFilter");
    }
    return lFilterInfo;
  }

  private void registerFilter( WebFilterInfo lFilterInfo, Filter pFilterInstance ) {
    for (String lPath : lFilterInfo.getURLPatterns()) {
      FilterHolder lFilter =
          contextHandler.addFilter(lFilterInfo.getFilterClass(), lPath, lFilterInfo.getDispatcherTypes());
      lFilter.setAsyncSupported(lFilterInfo.isAsyncSupported());
      lFilter.setDisplayName(lFilterInfo.getDisplayName());
      lFilter.setInitParameters(lFilterInfo.getInitParams());

      if (pFilterInstance != null) {
        lFilter.setFilter(pFilterInstance);
      }
      XFun.getTrace().info("Added servlet filter '" + lFilterInfo.getDisplayName() + "' under path '" + lPath
          + "' using implementation " + lFilterInfo.getFilterClass().getName());
    }
    filters.add(lFilterInfo);
  }

  public void start( ) {
    try {
      XFun.getTrace().info("JEAF Fast Lane is starting web container '" + name + "'");
      server.start();
    }
    catch (Exception e) {
      throw new JEAFSystemException(FastLaneMessages.WEB_CONTAINER_START_FAILED, e, name);
    }
  }

  public void join( ) {
    try {
      server.join();
    }
    catch (InterruptedException e) {
      XFun.getTrace().warn("Caught InterruptedException.", e);
      Thread.currentThread().interrupt();
    }
  }

  public void stop( ) {
    // Finished already accepted request in case of graceful shutdown.
    if (gracefulShutdownEnabled == true) {
      // Stop accepting new requests.
      gracefulShutdownFilter.shutdownRequested(true);

      boolean lFinished = this.isRequestProcessingFinished();
      if (lFinished == false) {
        this.waitUntilRequestProcessingFinished();
      }
      // No active requests
      else {
        XFun.getTrace().info("Stopping web container " + name + " immediately as there are no active requests.");
      }
    }

    // Really stop web container. If requests are still being processed or queued then they will be killed. This will
    // lead to client side error like connection refused.
    try {
      server.stop();
    }
    catch (Exception e) {
      throw new JEAFSystemException(FastLaneMessages.WEB_CONTAINER_STOP_FAILED, e, name);
    }
  }

  private void waitUntilRequestProcessingFinished( ) {
    // Calculate sleep time.
    int lSleepTime = Math.max(50, gracefulShutdownTimeout / 10);

    long lStart = System.currentTimeMillis();

    boolean lFinished = false;
    while (lFinished == false) {
      // Recalculate active threads and queue size.
      int lActiveThreads = this.getActiveRequests();
      XFun.getTrace().info("Currently " + lActiveThreads + " request(s) are processed or wait to be processed.");

      // Wait until all active requests are processed.
      try {
        Thread.sleep(lSleepTime);
      }
      catch (InterruptedException e) {
        XFun.getTrace().warn("Caught InterruptedException.", e);
        Thread.currentThread().interrupt();
      }

      // Calculate overall duration of grace period so far.
      long lWaitingDuration = System.currentTimeMillis() - lStart;

      // Check if we still need to wait until all requests are processed
      if (lWaitingDuration > gracefulShutdownTimeout || this.isRequestProcessingFinished() == true) {
        lFinished = true;
      }
      else {
        lFinished = false;
      }
    }

    // Grace period exceeded but still not all requests are processed. Now we will stop the the hard way.
    if (this.isRequestProcessingFinished() == false) {
      XFun.getTrace()
          .warn("Grace period of " + gracefulShutdownTimeout
              + "ms exceeded. Stopping web container even though not all requests were processed yet. "
              + this.getActiveRequests() + " requests are still waiting to be processed.");
    }
  }

  /**
   * Method checks if there are still request that are currently processed or are waiting to be processed.
   * 
   * @return boolean Method returns true if there are still processes that wait to be processed.
   */
  private boolean isRequestProcessingFinished( ) {
    // Active requests tells us how many requests are currently processed.
    return this.getActiveRequests() <= 0;
  }

  /**
   * Method returns the current status of the thread pool of this web container. Please be aware that the thread pools
   * status is usually very dynamic.
   * 
   * @return {@link WebContainerStateInfo} Thread pool status of this web container. The method never returns null.
   */
  public WebContainerStateInfo getState( ) {
    String lThreadPrefix = threadPool.getName();
    WebContainerState lStatus = this.getWebContainerStatus(server);
    int lThreadPoolSize = threadPool.getThreads();
    int lMinThreads = threadPool.getMinThreads();
    int lMaxThreads = threadPool.getMaxThreads();
    int lActiveRequests = this.getActiveRequests();
    int lQueueSize = threadPool.getQueueSize();
    return new WebContainerStateInfo(name, lStatus, lThreadPrefix, lThreadPoolSize, lMinThreads, lMaxThreads,
        lActiveRequests, maxQueueSize, lQueueSize);
  }

  /**
   * Method returns the liveness state of the web container. The liveness states defines if the internal state of the
   * container is valid. If liveness is broken then it means that the web container needs a restart to recover.
   * 
   * @return {@link HealthInfo} Object describung the liveness of the web container. The method never returns null.
   * 
   */
  public HealthInfo getLivenessState( ) {
    // Really resolve liveness state.
    HealthInfoState lHealthState;
    WebContainerState lWebContainerStatus = this.getWebContainerStatus(server);
    switch (lWebContainerStatus) {
      case RUNNING:
      case STARTING:
      case STOPPING:
        lHealthState = HealthInfoState.UP;
        break;

      default:
        lHealthState = HealthInfoState.OUT_OF_SERVICE;
    }

    return new HealthInfo("Liveness-Probe", lHealthState);
  }

  public HealthInfo getReadinessState( ) {
    // Really resolve liveness state.
    HealthInfoState lHealthState;
    WebContainerState lWebContainerStatus = this.getWebContainerStatus(server);
    switch (lWebContainerStatus) {
      case RUNNING:
        lHealthState = HealthInfoState.UP;
        break;

      default:
        lHealthState = HealthInfoState.OUT_OF_SERVICE;
    }

    return new HealthInfo("Readiness-Probe", lHealthState);
  }

  @Override
  public int getActiveRequests( ) {

    int lActiveRequest;
    if (gracefulShutdownFilter != null) {
      lActiveRequest = gracefulShutdownFilter.getActiveRequestCounter();
    }
    else {
      lActiveRequest = -1;
    }
    return lActiveRequest;
  }

  private WebContainerState getWebContainerStatus( Server pServer ) {
    WebContainerState lStatus;
    if (pServer.isStarting()) {
      lStatus = WebContainerState.STARTING;
    }
    else if (pServer.isStarted()) {
      lStatus = WebContainerState.RUNNING;
    }
    else if (pServer.isStopping()) {
      lStatus = WebContainerState.STOPPING;
    }
    else if (pServer.isStopped()) {
      lStatus = WebContainerState.STOPPED;
    }
    else if (pServer.isFailed()) {
      lStatus = WebContainerState.FAILED;
    }
    else {
      Assert.internalError("Jetty instance has unexpected state " + pServer.getState());
      lStatus = null;
    }
    return lStatus;
  }

  @Override
  public int getCurrentThreadPoolSize( ) {
    return threadPool.getThreads();
  }

  @Override
  public int getMinThreads( ) {
    return threadPool.getMinThreads();
  }

  @Override
  public int getMaxThreads( ) {
    return threadPool.getMaxThreads();
  }

  @Override
  public int getMaxQueueSize( ) {
    return threadPool.getMaxQueueSize();
  }

  @Override
  public int getQueueSize( ) {
    return threadPool.getQueueSize();
  }

  /**
   * Builder can be use to create new instances of a web container.
   */
  public static class Builder {
    /**
     * Static method to create new builder instance.
     */
    public static Builder newBuilder( ) {
      return new Builder();
    }

    /**
     * Initialize object.
     */
    private Builder( ) {
      // Nothing to do.
    }

    private String name = "WebContainer";

    private String contextPath = FastLaneConfig.DEFAULT_CONTEXT_PATH;

    private String threadPoolName = "worker";

    private int port = FastLaneConfig.DEFAULT_PORT;

    private int minThreads = FastLaneConfig.DEFAULT_WORKLOAD_MIN_THREADS;

    private int maxThreads = FastLaneConfig.DEFAULT_WORKLOAD_MAX_THREADS;

    private int threadIdleTimeout = FastLaneConfig.DEFAULT_THREAD_IDLE_TIMEOUT;

    private int httpIOIdleTimeout = FastLaneConfig.DEFAULT_HTTP_IO_IDLE_TIMEOUT;

    private boolean sendServerVersion = FastLaneConfig.DEFAULT_SEND_SERVER_VERSION;

    private boolean sendXPoweredBy = FastLaneConfig.DEFAULT_SEND_X_POWERED_BY;

    private boolean gracefulShutdownEnabled = FastLaneConfig.DEFAULT_GRACEFUL_SHUTDOWN;

    private int gracefulShutdownTimeout = FastLaneConfig.DEFAULT_GRACEFUL_SHUTDOWN_TIMEOUT;

    private int initialQueueSize = FastLaneConfig.DEFAULT_INITIAL_QUEUE_SIZE;

    private int maxQueueSize = FastLaneConfig.DEFAULT_MAX_QUEUE_SIZE;

    private int queueGrowSize = FastLaneConfig.DEFAULT_QUEUE_GROW_SIZE;

    private int inputBufferSize = FastLaneConfig.DEFAULT_INPUT_BUFFER_SIZE;

    private int outputBufferSize = FastLaneConfig.DEFAULT_OUTPUT_BUFFER_SIZE;

    public Builder setName( String pName ) {
      name = pName;
      return this;
    }

    public Builder setContextPath( String pContextPath ) {
      contextPath = pContextPath;
      return this;
    }

    public Builder setThreadPoolName( String pThreadPoolName ) {
      threadPoolName = pThreadPoolName;
      return this;
    }

    public Builder setPort( int pPort ) {
      port = pPort;
      return this;
    }

    public Builder setMinThreads( int pMinThreads ) {
      minThreads = pMinThreads;
      return this;
    }

    public Builder setMaxThreads( int pMaxThreads ) {
      maxThreads = pMaxThreads;
      return this;
    }

    public Builder setInitialQueueSize( int pInitialQueueSize ) {
      initialQueueSize = pInitialQueueSize;
      return this;
    }

    public Builder setMaxQueueSize( int pMaxQueueSize ) {
      maxQueueSize = pMaxQueueSize;
      return this;
    }

    public Builder setQueueGrowSize( int pQueueGrowSize ) {
      queueGrowSize = pQueueGrowSize;
      return this;
    }

    public Builder setInputBufferSize( int pInputBufferSize ) {
      inputBufferSize = pInputBufferSize;
      return this;
    }

    public Builder setOutputBufferSize( int pOutputBufferSize ) {
      outputBufferSize = pOutputBufferSize;
      return this;
    }

    public Builder setThreadIdleTimeout( int pThreadIdleTimeout ) {
      threadIdleTimeout = pThreadIdleTimeout;
      return this;
    }

    public Builder setHTTPIOIdleTimeout( int pHttpIOIdleTimeout ) {
      httpIOIdleTimeout = pHttpIOIdleTimeout;
      return this;
    }

    public Builder setSendServerVersion( boolean pSendServerVersion ) {
      sendServerVersion = pSendServerVersion;
      return this;
    }

    public Builder setSendXPoweredBy( boolean pSendXPoweredBy ) {
      sendXPoweredBy = pSendXPoweredBy;
      return this;
    }

    public Builder setGracefulShutdownEnabled( boolean pGracefulShutdownEnabled ) {
      gracefulShutdownEnabled = pGracefulShutdownEnabled;
      return this;
    }

    public Builder setGracefulShutdownTimeout( int pGracefulShutdownTimeout ) {
      gracefulShutdownTimeout = pGracefulShutdownTimeout;
      return this;
    }

    public WebContainer build( ) {
      return new WebContainer(this);
    }
  }
}
