/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletResponse;

import com.anaptecs.jeaf.tools.api.http.HTTPStatusCode;
import com.anaptecs.jeaf.xfun.api.XFun;

@WebFilter(value = "/*", asyncSupported = true, displayName = "Error Handler Filter")
public class ErrorHandlerFilter implements Filter {
  private final FastLaneServer server;

  private final boolean exitOnError;

  private final boolean traceUnhandledRuntimeErrors;

  private final boolean exitOnRuntimeException;

  private final boolean traceUnhandledRuntimeExceptions;

  public ErrorHandlerFilter( FastLaneServer pServer, boolean pExitOnError, boolean pTraceUnhandledRuntimeErrors,
      boolean pExitOnRuntimeException, boolean pTraceUnhandledExceptions ) {
    server = pServer;
    exitOnError = pExitOnError;
    traceUnhandledRuntimeErrors = pTraceUnhandledRuntimeErrors;
    exitOnRuntimeException = pExitOnRuntimeException;
    traceUnhandledRuntimeExceptions = pTraceUnhandledExceptions;
  }

  @Override
  public void init( FilterConfig pFilterConfig ) throws ServletException {
    // Nothing to do
  }

  @Override
  public void doFilter( ServletRequest pRequest, ServletResponse pResponse, FilterChain pChain )
    throws IOException, ServletException {

    // Delegate incoming request to rest of filter chain. This filter is only interested in may be occuring exceptions
    try {
      pChain.doFilter(pRequest, pResponse);
    }
    catch (RuntimeException e) {
      this.handleUncaughtException(pResponse, e, exitOnRuntimeException, traceUnhandledRuntimeExceptions);
    }
    catch (Error e) {
      this.handleUncaughtException(pResponse, e, exitOnError, traceUnhandledRuntimeErrors);
    }
  }

  private void handleUncaughtException( ServletResponse pResponse, Throwable pThrowable, boolean pRequestShutdown,
      boolean pTraceException ) {

    // Set http status code in response.
    ((HttpServletResponse) pResponse).setStatus(HTTPStatusCode.INTERNAL_SERVER_ERROR.getCode());

    // Trace uncaught exception.
    if (pTraceException == true) {
      XFun.getTrace().error("Caught unhandled exception in servlet filter chain.", pThrowable);
    }

    // Shutdown server due to exception.
    if (pRequestShutdown == true) {
      XFun.getTrace().error("Shutting do JEAF Fast Lane server due to uncaught exception.");
      server.requestShutdown(250, -1);
    }
  }

  @Override
  public void destroy( ) {
    // Nothing to do
  }
}
