/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.anaptecs.jeaf.tools.api.http.HTTPStatusCode;
import com.anaptecs.jeaf.xfun.api.XFun;

@WebFilter(value = "/*", asyncSupported = true, displayName = "Graceful Shutdown Filter")
public class GracefulShutdownFilter implements Filter {
  /**
   * Field defines if we are in shutdown mode.
   */
  private boolean shutdownRequested = false;

  /**
   * Counter is used to track the amount of requests that are currently active
   */
  private AtomicInteger activeRequestCounter = new AtomicInteger(0);

  /**
   * Method returns if we are in the shutdown mode.
   * 
   * @return boolean Method returns true if a shutdown was requested and false otherwise.
   */
  public boolean isShutdownRequested( ) {
    return shutdownRequested;
  }

  /**
   * Method can be used to switch the filter into shutdown mode. Shutdown mode means that all arriving requests will be
   * rejected with HTTP status code SERVICE_UNAVAILABLE (503)
   * 
   * @param pShutdownRequested Parameter will switch filter to shutdown mode in case that it is true and will disable
   * shutdown mode if it is false.
   */
  public void shutdownRequested( boolean pShutdownRequested ) {
    shutdownRequested = pShutdownRequested;
  }

  /**
   * Method increments the request counter by one.
   * 
   * @return int Current value of request counter after it was incremented.
   */
  public int incrementActiveRequestCounter( ) {
    return activeRequestCounter.incrementAndGet();
  }

  /**
   * Method decrements the request counter by one.
   * 
   * @return int Current value of request counter after it was decremented.
   */
  public int decrementActiveRequestCounter( ) {
    return activeRequestCounter.decrementAndGet();
  }

  /**
   * Method returns the current amount of active requests.
   * 
   * @return int Current amount of active requests.
   */
  public int getActiveRequestCounter( ) {
    return activeRequestCounter.get();
  }

  /**
   * Initialize filter.
   */
  @Override
  public void init( FilterConfig pFilterConfig ) throws ServletException {
    // Nothing to do.
  }

  /**
   * Implementation of the concrete filtering. In our case here we do the following things:
   * <ul>
   * <li>In normal mode we just count the amount of active requests</li>
   * <li>In shutdown mode reject incoming requests as we try to do a graceful shutdown</li>
   * </ul>
   */
  @Override
  public void doFilter( ServletRequest pRequest, ServletResponse pResponse, FilterChain pChain )
    throws IOException, ServletException {

    // Web container is active
    if (this.isShutdownRequested() == false) {
      try {
        this.incrementActiveRequestCounter();
        pChain.doFilter(pRequest, pResponse);
      }
      finally {
        this.decrementActiveRequestCounter();
      }
    }
    else {
      HttpServletRequest lHttpServletRequest = (HttpServletRequest) pRequest;
      HttpServletResponse lHttpServletResponse = (HttpServletResponse) pResponse;
      XFun.getTrace().warn(
          "Rejecting request " + lHttpServletRequest.getServletPath() + " as server is currently in shutdown process.");
      lHttpServletResponse.setStatus(HTTPStatusCode.SERVICE_UNAVAILABLE.getCode());
      lHttpServletResponse.getWriter().write("Server is shutting down. No further new requests will be accepted.");
    }
  }

  /**
   * Method will be called before the filter is finally deleted.
   */
  @Override
  public void destroy( ) {
    // Nothing to do.
  }
}
