/**
 * Copyright 2004 - 2021 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.fastlane.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.servlet.DispatcherType;
import javax.servlet.Filter;
import javax.servlet.annotation.WebFilter;
import javax.servlet.annotation.WebInitParam;

import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.xfun.api.XFun;
import com.anaptecs.jeaf.xfun.api.checks.Check;

public class WebFilterInfo {
  private final Class<? extends Filter> filterClass;

  private final String displayName;

  private final boolean asyncSupported;

  private final EnumSet<DispatcherType> dispatcherTypes;

  private final Map<String, String> initParams;

  private final SortedSet<String> urlPatterns;

  private WebFilterInfo( Builder pBuilder ) {
    // Check parameters
    Check.checkInvalidParameterNull(pBuilder.filterClass, "pBuilder.filterClass");

    filterClass = pBuilder.filterClass;
    displayName = pBuilder.displayName;
    asyncSupported = pBuilder.asyncSupported;
    dispatcherTypes = EnumSet.copyOf(Arrays.asList(pBuilder.dispatcherTypes));
    initParams = Collections.unmodifiableMap(new HashMap<>(pBuilder.initParams));
    urlPatterns = Collections.unmodifiableSortedSet(new TreeSet<>(pBuilder.urlPatterns));
  }

  public Class<? extends Filter> getFilterClass( ) {
    return filterClass;
  }

  public String getDisplayName( ) {
    return displayName;
  }

  public boolean isAsyncSupported( ) {
    return asyncSupported;
  }

  public EnumSet<DispatcherType> getDispatcherTypes( ) {
    return dispatcherTypes;
  }

  public Map<String, String> getInitParams( ) {
    return initParams;
  }

  public SortedSet<String> getURLPatterns( ) {
    return urlPatterns;
  }

  static class Builder {
    private Class<? extends Filter> filterClass;

    private String displayName;

    private boolean asyncSupported;

    private DispatcherType[] dispatcherTypes;

    private final Map<String, String> initParams = new HashMap<>();

    private final Set<String> urlPatterns = new HashSet<>();

    static Builder newBuilder( ) {
      return new Builder();
    }

    private Builder( ) {
      // Nothing to do.
    }

    public Builder setFilterClass( Class<? extends Filter> pFilterClass ) {
      filterClass = pFilterClass;
      return this;
    }

    public Builder setFromFilterAnnotation( WebFilter pAnnotation ) {
      String lDisplayName = pAnnotation.displayName();
      if (Tools.getStringTools().isRealString(lDisplayName) == true) {
        displayName = lDisplayName;
      }
      else {
        displayName = filterClass.getSimpleName();
      }
      asyncSupported = pAnnotation.asyncSupported();
      dispatcherTypes = pAnnotation.dispatcherTypes();

      for (WebInitParam lParam : pAnnotation.initParams()) {
        initParams.put(lParam.name(), lParam.value());
      }

      boolean lURLPatternAdded = false;
      for (String lPattern : pAnnotation.value()) {
        urlPatterns.add(lPattern);
        lURLPatternAdded = true;
      }

      for (String lPattern : pAnnotation.urlPatterns()) {
        urlPatterns.add(lPattern);
        lURLPatternAdded = true;
      }
      if (lURLPatternAdded == false) {
        urlPatterns.add("/*");
      }

      if (pAnnotation.servletNames().length > 0) {
        XFun.getTrace().warn("JEAF Fast Lane does not support mapping from filter to servlets.");
      }
      return this;
    }

    WebFilterInfo build( ) {
      return new WebFilterInfo(this);
    }
  }

}
