/**
 * Copyright 2004 - 2022 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.jersey.impl.time;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.ws.rs.ext.ParamConverter;

/**
 * Class implements a JAX-RS parameter converter. Parameter converters are used in JAX-RS to convert REST parameters
 * from their Java object representation to Strings and vice versa.
 * 
 * This class implements a parameter converter for {@link Calendar} to {@link String} and back.
 * 
 * @author JEAF Development Team
 */
public class CalendarParamConverter implements ParamConverter<Calendar> {
  /**
   * @see ParamConverter#fromString(String)
   */
  @Override
  public Calendar fromString( String pValue ) {
    // Check parameter for null.
    if (pValue != null) {
      SimpleDateFormat lDateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
      lDateTimeFormatter.setLenient(true);
      try {
        Date lParse = lDateTimeFormatter.parse(pValue);
        Calendar lCalendar = Calendar.getInstance();
        lCalendar.setTime(lParse);
        return lCalendar;
      }
      catch (ParseException e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }

  /**
   * @see ParamConverter#toString(Object)
   */
  @Override
  public String toString( Calendar pValue ) {
    // Check parameter for null
    if (pValue != null) {
      SimpleDateFormat lDateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
      return lDateTimeFormatter.format(pValue.getTime());
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }
}
