/**
 * Copyright 2004 - 2022 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.jersey.impl.time;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

import javax.ws.rs.ext.ParamConverter;

/**
 * Class implements a JAX-RS parameter converter. Parameter converters are used in JAX-RS to convert REST parameters
 * from their Java object representation to Strings and vice versa.
 * 
 * This class implements a parameter converter for {@link LocalDate} to {@link String} and back.
 * 
 * @author JEAF Development Team
 */
public class LocalDateParamConverter implements ParamConverter<LocalDate> {
  /**
   * @see ParamConverter#fromString(String)
   */
  @Override
  public LocalDate fromString( String pValue ) {
    // Check parameter for null.
    if (pValue != null) {
      try {
        return LocalDate.parse(pValue);
      }
      catch (DateTimeParseException e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }

  /**
   * @see ParamConverter#toString(Object)
   */
  @Override
  public String toString( LocalDate pValue ) {
    // Check parameter for null.
    if (pValue != null) {
      return DateTimeFormatter.ISO_DATE.format(pValue);
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }
}
