/**
 * Copyright 2004 - 2022 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.jersey.impl.time;

import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

import javax.ws.rs.ext.ParamConverter;

/**
 * Class implements a JAX-RS parameter converter. Parameter converters are used in JAX-RS to convert REST parameters
 * from their Java object representation to Strings and vice versa.
 * 
 * This class implements a parameter converter for {@link OffsetDateTime} to {@link String} and back.
 * 
 * @author JEAF Development Team
 */
public class OffsetDateTimeParamConverter implements ParamConverter<OffsetDateTime> {
  /**
   * @see ParamConverter#fromString(String)
   */
  @Override
  public OffsetDateTime fromString( String pValue ) {
    // Check parameter for null.
    if (pValue != null) {
      try {
        return OffsetDateTime.parse(pValue);
      }
      catch (DateTimeParseException e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }

  /**
   * @see ParamConverter#toString(Object)
   */
  @Override
  public String toString( OffsetDateTime pValue ) {
    // Check parameter for null
    if (pValue != null) {
      return DateTimeFormatter.ISO_DATE_TIME.format(pValue);
    }
    // According to contract defined in base class the parameter must not be null.
    else {
      throw new IllegalArgumentException("Parameter 'pValue' must not be null.");
    }
  }
}
