/**
 * Copyright 2004 - 2019 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.maven;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;

import com.anaptecs.jeaf.core.annotations.ComponentFactory;
import com.anaptecs.jeaf.core.annotations.CoreConfig;
import com.anaptecs.jeaf.core.annotations.CoreFactory;
import com.anaptecs.jeaf.core.annotations.EJBClientServiceChannelConfig;
import com.anaptecs.jeaf.core.annotations.EJBContainerConfig;
import com.anaptecs.jeaf.core.annotations.JEAFActivityImpl;
import com.anaptecs.jeaf.core.annotations.LifecycleManagerFactory;
import com.anaptecs.jeaf.core.annotations.SchedulerCredentials;
import com.anaptecs.jeaf.core.annotations.SchedulingConfig;
import com.anaptecs.jeaf.core.annotations.SecurityConfig;
import com.anaptecs.jeaf.core.annotations.ServiceProviderFactory;
import com.anaptecs.jeaf.spi.persistence.annotations.PersistenceConfig;
import com.anaptecs.jeaf.tools.annotations.EncryptionToolsConfig;
import com.anaptecs.jeaf.tools.annotations.ToolsConfig;
import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.tools.api.performance.Stopwatch;
import com.anaptecs.jeaf.tools.api.performance.TimePrecision;
import com.anaptecs.jeaf.workload.annotations.ElasticWorkloadConfig;
import com.anaptecs.jeaf.workload.annotations.PipelineConfig;
import com.anaptecs.jeaf.workload.annotations.StaticWorkloadConfig;
import com.anaptecs.jeaf.workload.annotations.WorkloadManagementConfig;
import com.anaptecs.jeaf.workload.api.Workload;
import com.anaptecs.jeaf.xfun.annotations.AppInfo;
import com.anaptecs.jeaf.xfun.annotations.ConfigurationProviderConfig;
import com.anaptecs.jeaf.xfun.annotations.DatatypeConverterImpl;
import com.anaptecs.jeaf.xfun.annotations.MessageResource;
import com.anaptecs.jeaf.xfun.annotations.RuntimeInfo;
import com.anaptecs.jeaf.xfun.annotations.StartupInfoConfig;
import com.anaptecs.jeaf.xfun.annotations.StartupInfoWriterImpl;
import com.anaptecs.jeaf.xfun.annotations.TraceConfig;
import com.anaptecs.jeaf.xfun.annotations.TraceObjectFormatter;
import com.anaptecs.jeaf.xfun.annotations.XFunConfig;
import com.anaptecs.jeaf.xfun.impl.info.VersionInfoHelper;

import io.github.classgraph.ScanResult;

@Mojo(
    name = "GenerateJEAFConfig",
    defaultPhase = LifecyclePhase.PROCESS_TEST_CLASSES,
    threadSafe = true,
    requiresDependencyResolution = ResolutionScope.TEST)

public class GenerateConfigMojo extends AbstractMojo {
  /**
   * Name of the base package under which all X-Fun configuration files are located.
   */
  public static final String JEAF_META_INF_BASE_PACKAGE = "META-INF/JEAF/XFun";

  /**
   * Reference to Maven project. Reference will be injected by Maven and can not be configured via POM.
   */
  @Parameter(defaultValue = "${project}", required = true, readonly = true)
  private MavenProject project;

  /**
   * Configuration parameter defines the output directory to which the generated JEAF configuration files should be
   * written. The plugin will analyze therefore all dependencies that are not of scope "test".
   * 
   * At least one of "resourceGenDirectory" or "testResourceGenDirectory" has to be set.
   */
  @Parameter(name = "resourceGenDirectory", required = false)
  private String resourceGenDirectory;

  /**
   * Configuration parameter defines the output directory to which the generated JEAF configuration files for test scope
   * should be written. The plugin will analyze therefore all dependencies including also those of scope "test".
   * 
   * At least one of "resourceGenDirectory" or "testResourceGenDirectory" has to be set.
   */
  @Parameter(name = "testResourceGenDirectory", required = false)
  private String testResourceGenDirectory;

  /**
   * This mojo also supports writing file to target directory. There the target directory has to be passed through this
   * configuration parameter.
   */
  @Parameter(name = "targetDirectory", required = false)
  private String targetDirectory;

  /**
   * This mojo also supports writing file to target test directory. There the target directory has to be passed through
   * this configuration parameter.
   */
  @Parameter(name = "testTargetDirectory", required = false)
  private String testTargetDirectory;

  /**
   * Parameter can be used to include classes that match some specific pattern. The pattern matching mechanism is rather
   * simple. Thus only the following variants are supported:
   * <ul>
   * <li><code>com.anaptecs.jeaf.*</code> meaning that all classes whose name starts with "com.anaptecs.jeaf." will be
   * included</li>
   * <li><code>*.MessageConstant</code> meaning that all classes whose name ends with ".MessageConstant" will be
   * included</li>
   * <li><code>*.MessageConstant*</code> meaning that all classes whose name contains ".MessageConstant" will be
   * included</li>
   * <li><code>com.anaptecs.jeaf.components.MessageConstants</code> meaning that class with name
   * "com.anaptecs.jeaf.components.MessageConstants" will be included</li>
   * </ul>
   * 
   * Either configuration parameter "includes" or "excludes" can be used but not both.
   */
  @Parameter(name = "includes")
  private List<String> includes = new ArrayList<>(0);

  /**
   * Parameter can be used to exclude classes that match some specific pattern. The pattern matching mechanism is rather
   * simple. Thus only the following variants are supported:
   * <ul>
   * <li><code>com.anaptecs.jeaf.*</code> meaning that all classes whose name starts with "com.anaptecs.jeaf." will be
   * excluded</li>
   * <li><code>*.MessageConstant</code> meaning that all classes whose name ends with ".MessageConstant" will be
   * excluded</li>
   * <li><code>*.MessageConstant*</code> meaning that all classes whose name contains ".MessageConstant" will be
   * excluded</li>
   * <li><code>com.anaptecs.jeaf.components.MessageConstants</code> meaning that class with name
   * "com.anaptecs.jeaf.components.MessageConstants" will be excluded</li>
   * </ul>
   * 
   * Either configuration parameter "includes" or "excludes" can be used but not both.
   */
  @Parameter(name = "excludes")
  private List<String> excludes = new ArrayList<>();

  @Parameter(defaultValue = "false", required = false)
  private boolean cleanMetaInfDirectory;

  @Parameter(defaultValue = "false", required = false)
  private boolean cleanOnly;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectMessageResources;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectComponentFactories;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectServiceProviderFactories;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectActivities;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectAppInfo;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectRuntimeInfo;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectConfigurationProviderConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectTraceConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectTraceObjectFormatter;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectXFunConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectLifecycleManagerFactory;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectDatatypeConverterImpls;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectToolsConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectStartupInfoConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectStartupInfoWriterImpl;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectCoreFactory;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectCoreConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectSchedulingConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectSecurityConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectSchedulerCredentials;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectWorkloadManagerConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectWorkloadMappings;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectPipelineConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectEJBClientServiceChannelConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectEJBContainerConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean detectPersistenceConfig;

  @Parameter(defaultValue = "true", required = false)
  private boolean createVersionInfo;

  /**
   * When using explicit configuration detection then you explicitly have to define which configurations should be
   * detected.
   */
  @Parameter
  private ExplicitConfig explicitConfigurationDetection;

  @Parameter(defaultValue = "${project.version}", required = true, readonly = true)
  private String version;

  @Parameter(defaultValue = AppInfo.VERSION_INFO_PATH, required = false)
  private String versionInfoPath;

  private NameFilter filter;

  /**
   * Method starts execution of this plugin.
   */
  @Override
  public void execute( ) throws MojoExecutionException, MojoFailureException {
    try {
      // Create new name filter
      filter = new NameFilter(includes, excludes);

      this.configurePlugin();

      // Run clean up
      if (cleanMetaInfDirectory == true) {
        this.cleanMetaInfDirectories();
      }

      // Start generation of config files.
      if (cleanOnly == false) {
        if (resourceGenDirectory != null || testResourceGenDirectory != null) {
          // Scan classpath for standard resources (not test scope)
          if (resourceGenDirectory != null) {
            this.generateConfigs(false);
          }
          // Scan classpath for test scope.
          if (testResourceGenDirectory != null) {
            this.generateConfigs(true);
          }
        }
        // We need at least one output directory for the configuration files.
        else {
          throw new MojoFailureException(
              "'resourceGenDirectory' or 'testRessourceGenDirectory' or both must be configured. Please check plugins configuration.");
        }
      }
    }
    catch (IOException e) {
      throw new MojoFailureException(e.getMessage(), e);
    }
    catch (DependencyResolutionRequiredException e) {
      throw new MojoFailureException(e.getMessage(), e);
    }
    catch (RuntimeException | Error e) {
      this.getLog().error(e);
    }
  }

  private void configurePlugin( ) {
    // Configuration detection is configured to work in explicit mode.
    if (explicitConfigurationDetection != null) {
      this.disableAll();

      // X-Fun configuration should be detected.
      if (explicitConfigurationDetection.detectXFunConfigs == true) {
        // X-Fun configurations
        createVersionInfo = true;
        detectAppInfo = true;
        detectConfigurationProviderConfig = true;
        detectDatatypeConverterImpls = true;
        detectMessageResources = true;
        detectRuntimeInfo = true;
        detectStartupInfoConfig = true;
        detectStartupInfoWriterImpl = true;
        detectTraceConfig = true;
        detectTraceObjectFormatter = true;
      }

      // Tools configuration should be detected.
      if (explicitConfigurationDetection.detectToolsConfigs == true) {
        // Tools configurations
        detectToolsConfig = true;
      }

      // Core configurations should be detected.
      if (explicitConfigurationDetection.detectCoreConfigs == true) {
        // Core configurations
        detectActivities = true;
        detectComponentFactories = true;
        detectCoreConfig = true;
        detectCoreFactory = true;
        detectEJBClientServiceChannelConfig = true;
        detectEJBContainerConfig = true;
        detectLifecycleManagerFactory = true;
        detectSchedulerCredentials = true;
        detectSchedulingConfig = true;
        detectSecurityConfig = true;
        detectServiceProviderFactories = true;
      }

      // Persistence configurations should be detected.
      if (explicitConfigurationDetection.detectPersistenceConfigs == true) {
        // Persistence configurations
        detectPersistenceConfig = true;
      }

      // Workload configurations should be detected.
      if (explicitConfigurationDetection.detectWorkloadConfigs == true) {
        // Workload configurations
        detectPipelineConfig = true;
        detectWorkloadManagerConfig = true;
        detectWorkloadMappings = true;
      }
    }
    // By default everything should be detected.
    else {
      // That's our default, so nothing to do.
    }
  }

  private void disableAll( ) {
    // X-Fun configurations
    createVersionInfo = false;
    detectAppInfo = false;
    detectConfigurationProviderConfig = false;
    detectDatatypeConverterImpls = false;
    detectMessageResources = false;
    detectRuntimeInfo = false;
    detectStartupInfoConfig = false;
    detectStartupInfoWriterImpl = false;
    detectTraceConfig = false;
    detectTraceObjectFormatter = false;

    // Tools configurations
    detectToolsConfig = false;

    // Core configurations
    detectActivities = false;
    detectComponentFactories = false;
    detectCoreConfig = false;
    detectCoreFactory = false;
    detectEJBClientServiceChannelConfig = false;
    detectEJBContainerConfig = false;
    detectLifecycleManagerFactory = false;
    detectSchedulerCredentials = false;
    detectSchedulingConfig = false;
    detectSecurityConfig = false;
    detectServiceProviderFactories = false;

    // Persistence configurations
    detectPersistenceConfig = false;

    // Workload configurations
    detectPipelineConfig = false;
    detectWorkloadManagerConfig = false;
    detectWorkloadMappings = false;
  }

  private void cleanMetaInfDirectories( ) {
    // Delete standard resource gen directory with all its content.
    if (resourceGenDirectory != null) {
      String lMetaInfDirectory = resourceGenDirectory + File.separatorChar + JEAF_META_INF_BASE_PACKAGE;
      File lDirectory = new File(lMetaInfDirectory);
      if (lDirectory.exists() == true) {
        this.getLog().info("Cleaning directory " + lMetaInfDirectory);
        Tools.getFileTools().deleteRecursive(lMetaInfDirectory);
      }
    }
    // Delete test resource gen directory with all its content.
    if (testResourceGenDirectory != null) {
      String lMetaInfDirectory = testResourceGenDirectory + File.separatorChar + JEAF_META_INF_BASE_PACKAGE;
      File lDirectory = new File(lMetaInfDirectory);
      if (lDirectory.exists() == true) {
        this.getLog().info("Cleaning directory " + lMetaInfDirectory);
        Tools.getFileTools().deleteRecursive(lMetaInfDirectory);
      }
    }
  }

  public void generateConfigs( boolean pTestScope ) throws IOException, DependencyResolutionRequiredException {
    // Scan all dependencies
    ScanResult lScanResult = null;
    AnnotationBasedConfigGenerator lGenerator = null;
    try {
      // Scan classpath of project
      // As scanning of classpath may be rather time consuming the start a stop watch to measure it.
      Stopwatch lStopwatch = Tools.getPerformanceTools().createStopwatch("Classpath scan took: ", TimePrecision.MILLIS);
      lStopwatch.start();

      // Resolve path of all dependencies.
      List<String> lClasspathPaths = this.getDependencyPaths(pTestScope);

      // Scan class path
      Scanner lScanner = new Scanner();
      lScanResult = lScanner.scan(lClasspathPaths);

      // Map<ClassInfo, ClassInfoList> lDependencyMap = lScanResult.getClassDependencyMap();
      // for (ClassInfo lNextClass : lDependencyMap.keySet()) {
      // this.getLog().info(lNextClass.getName() + ":");
      // ClassInfoList lDependentClasses = lDependencyMap.get(lNextClass);
      // for (ClassInfo lNext : lDependentClasses) {
      // this.getLog().info(" " + lNext.getName() + " " + lNext.getClasspathElementFile());
      // }
      // }

      lGenerator = new AnnotationBasedConfigGenerator(resourceGenDirectory, testResourceGenDirectory, targetDirectory,
          testTargetDirectory, lScanResult, getLog(), lClasspathPaths);

      // Write some trace about duration of classpath scan.
      lStopwatch.stopAndTrace();

      // Build list with all annotations that are supported.
      List<AnnotationProcessingConfig> lSupportedAnnotations = new ArrayList<>();

      // Annotation @MessageResource
      lSupportedAnnotations.add(new AnnotationProcessingConfig(MessageResource.class,
          MessageResource.MESSAGE_RESOURCES_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectMessageResources));

      // Annotation @ComponentFactory
      lSupportedAnnotations.add(new AnnotationProcessingConfig(ComponentFactory.class,
          ComponentFactory.COMPONENT_FACTORIES_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectComponentFactories));

      // Annotation @ServiceProviderFactory
      lSupportedAnnotations.add(new AnnotationProcessingConfig(ServiceProviderFactory.class,
          ServiceProviderFactory.SERVICE_PROVIDER_FACTORIES_PATH, ClassOccurrencePolicy.ALL_CLASSES,
          detectServiceProviderFactories));

      // Annotation @JEAFActivityImpl
      lSupportedAnnotations.add(new AnnotationProcessingConfig(JEAFActivityImpl.class,
          JEAFActivityImpl.ACTIVITIY_IMPL_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectActivities));

      // Annotation @AppInfo
      lSupportedAnnotations.add(new AnnotationProcessingConfig(AppInfo.class, AppInfo.APP_INFO_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectAppInfo));

      // Annotation @RuntimeInfo
      lSupportedAnnotations.add(new AnnotationProcessingConfig(RuntimeInfo.class, RuntimeInfo.RUNTIME_INFO_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectRuntimeInfo));

      // Annotation @ConfigurationProviderConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(ConfigurationProviderConfig.class,
          ConfigurationProviderConfig.CONFIGURATION_PROVIDER_CONFIG_PATH, ClassOccurrencePolicy.FIRST_CLASS,
          detectConfigurationProviderConfig));

      // Annotation @TraceConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(TraceConfig.class, TraceConfig.TRACE_CONFIG_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectTraceConfig));

      // Annotation @TraceObjectFormatter
      lSupportedAnnotations.add(
          new AnnotationProcessingConfig(TraceObjectFormatter.class, TraceObjectFormatter.TRACE_OBJECT_FORMATTER_PATH,
              ClassOccurrencePolicy.ALL_CLASSES, detectTraceObjectFormatter));

      // Annotation @XFunConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(XFunConfig.class, XFunConfig.XFUN_CONFIG_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectXFunConfig));

      // Annotation @LifecycleManagerFactory
      lSupportedAnnotations.add(new AnnotationProcessingConfig(LifecycleManagerFactory.class,
          LifecycleManagerFactory.LIFECYCLE_MANAGER_FACTORY_PATH, ClassOccurrencePolicy.FIRST_CLASS,
          detectLifecycleManagerFactory));

      // Annotation @DatatypeConverterImpl
      lSupportedAnnotations.add(
          new AnnotationProcessingConfig(DatatypeConverterImpl.class, DatatypeConverterImpl.DATATYPE_CONVERTERS_PATH,
              ClassOccurrencePolicy.ALL_CLASSES, detectDatatypeConverterImpls));

      // Annotation @ToolsConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(ToolsConfig.class, ToolsConfig.TOOLS_CONFIG_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectToolsConfig));

      // Annotation @EncryptionToolsConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(EncryptionToolsConfig.class,
          EncryptionToolsConfig.ENCRYPTION_TOOLS_CONFIG_PATH, ClassOccurrencePolicy.FIRST_CLASS, detectToolsConfig));

      // Annotation @StartupInfoConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(StartupInfoConfig.class,
          StartupInfoConfig.STARTUP_INFO_CONFIG_PATH, ClassOccurrencePolicy.FIRST_CLASS, detectStartupInfoConfig));

      // Annotation @StartupInfoWriterImpl
      lSupportedAnnotations.add(
          new AnnotationProcessingConfig(StartupInfoWriterImpl.class, StartupInfoWriterImpl.STARTUP_INFO_WRITERS_PATH,
              ClassOccurrencePolicy.ALL_CLASSES, detectStartupInfoWriterImpl));

      // Annotation @CoreFactory
      lSupportedAnnotations.add(new AnnotationProcessingConfig(CoreFactory.class, CoreFactory.JEAF_CORE_FACTORY_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectCoreFactory));

      // Annotation @CoreConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(CoreConfig.class, CoreConfig.JEAF_CORE_CONFIG_PATH,
          ClassOccurrencePolicy.FIRST_CLASS, detectCoreConfig));

      // Annotation @SchedulingConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(SchedulingConfig.class,
          SchedulingConfig.SCHEDULING_CONFIG_PATH, ClassOccurrencePolicy.FIRST_CLASS, detectSchedulingConfig));

      // Annotation @SecurityConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(SecurityConfig.class,
          SecurityConfig.SECURITY_CONFIG_PATH, ClassOccurrencePolicy.FIRST_CLASS, detectSecurityConfig));

      // Annotation @SchedulerCredentials
      lSupportedAnnotations.add(new AnnotationProcessingConfig(SchedulerCredentials.class,
          SchedulerCredentials.SCHEDULER_CREDENTIALS_CONFIG_PATH, ClassOccurrencePolicy.ONE_CLASS_ONLY,
          detectSchedulerCredentials));

      // Annotation @WorkloadManagementConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(WorkloadManagementConfig.class,
          Workload.WORKLOAD_MANAGER_RESOURCE_PATH, ClassOccurrencePolicy.ONE_CLASS_ONLY, detectWorkloadManagerConfig));

      // Annotation @PipelineConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(PipelineConfig.class, Workload.PIPELINES_RESOURCE_PATH,
          ClassOccurrencePolicy.ALL_CLASSES, detectPipelineConfig));

      // Annotation @StaticWorkloadConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(StaticWorkloadConfig.class,
          Workload.STATIC_WORKLOADS_RESOURCE_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectWorkloadMappings));

      // Annotation @ElasticWorkloadConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(ElasticWorkloadConfig.class,
          Workload.ELASTIC_WORKLOADS_RESOURCE_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectWorkloadMappings));

      // Annotation @EJBClientServiceChannelConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(EJBClientServiceChannelConfig.class,
          EJBClientServiceChannelConfig.EJB_CLIENT_SERVICE_CHANNEL_CONFIG_RESOURCE_PATH,
          ClassOccurrencePolicy.ONE_CLASS_ONLY, detectEJBClientServiceChannelConfig));

      // Annotation @EJBContainerConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(EJBContainerConfig.class,
          EJBContainerConfig.EJB_CONTAINER_CONFIG_RESOURCE_PATH, ClassOccurrencePolicy.ONE_CLASS_ONLY,
          detectEJBContainerConfig));

      // Annotation @PersistenceConfig
      lSupportedAnnotations.add(new AnnotationProcessingConfig(PersistenceConfig.class,
          PersistenceConfig.PERSISTENCE_CONFIG_PATH, ClassOccurrencePolicy.ALL_CLASSES, detectPersistenceConfig));

      // Generate config files for all annotations
      for (AnnotationProcessingConfig lAnnotationProcessingConfig : lSupportedAnnotations) {
        // If annotation detection is enabled then we run the generator.
        if (lAnnotationProcessingConfig.isEnabled() == true) {
          lGenerator.generate(lAnnotationProcessingConfig, filter, pTestScope);
        }
      }

      // Create version info file
      if (createVersionInfo == true) {
        Log lLog = this.getLog();
        lLog.info(" ");
        lLog.info("Version-Info:");

        Date lBuildStartTime = project.getProjectBuildingRequest().getBuildStartTime();
        String lCreationDate = Tools.getDateTools().toTimestampString(lBuildStartTime);
        List<String> lProperties = new ArrayList<>();
        lProperties.add(VersionInfoHelper.VERSION + "=" + version);
        lProperties.add(VersionInfoHelper.CREATION_DATE + "=" + lCreationDate);

        for (String lValue : lProperties) {
          lLog.info("    " + lValue);
        }
        lLog.info(" ");

        // Write config file
        lGenerator.writeConfigFile(lProperties, versionInfoPath, pTestScope);
      }
    }
    finally {
      if (lScanResult != null) {
        lScanResult.close();
      }
    }
  }

  /**
   * Method resolves all dependencies of the project under which the mojo is executed.
   * 
   * @param pTestScope Parameter defines if all compile and runtime dependencies should be resolved only or also test
   * dependencies in addition.
   * @return {@link List} List with paths of all dependencies. These paths represent the classpath entries. The method
   * never returns null.
   * @throws IOException
   * @throws DependencyResolutionRequiredException
   */
  private List<String> getDependencyPaths( boolean pTestScope )
    throws IOException, DependencyResolutionRequiredException {
    // Resolve all dependencies for current project. Therefore we have to distinguish between test and normal scope.
    List<String> lDependencyPaths;

    // We are in test scope. In this case we have to use the test class path.
    if (pTestScope == true) {
      lDependencyPaths = project.getTestClasspathElements();
    }
    // In standard scope we can resolve our classpath through the artifacts.
    else {
      List<Artifact> lArtifacts = new ArrayList<>(project.getArtifacts());
      lDependencyPaths = new ArrayList<>(lArtifacts.size());

      // In addition we also have to add the current project itself.
      Artifact lProjectArtifact = project.getArtifact();
      if (lProjectArtifact != null) {
        lArtifacts.add(0, lProjectArtifact);
      }

      // Process all artifacts.
      for (Artifact lArtifact : lArtifacts) {
        // In case that we are not in test scope then we have to filter test dependencies
        String lScope = lArtifact.getScope();
        if (pTestScope == false) {
          if ("test".equalsIgnoreCase(lScope) == true) {
            lArtifact = null;
          }
        }

        // If artifact was not filtered out then we will now put it into list of dependencies.
        if (lArtifact != null) {
          File lFile = lArtifact.getFile();
          if (lFile != null) {
            String lArtifactPath = lFile.getCanonicalPath();
            lDependencyPaths.add(lArtifactPath);
          }
          else {
            this.getLog().info("Artifact " + lArtifact.toString() + " has no local file.");
          }
        }
      }
    }

    // Write log about resolved dependencies
    this.getLog().info("Detected the following Maven dependencies:");
    for (String lNextclasspathEntry : lDependencyPaths) {
      this.getLog().info("    " + lNextclasspathEntry);
    }

    // Return path of all classpath entries.
    return lDependencyPaths;
  }
}
