/**
 * Copyright 2004 - 2019 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.maven;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Class implements a name filter that supports includes and exclude expressions.
 * 
 * @author JEAF Development Team
 * @version JEAF Release 1.3.1
 */
public class NameFilter {
  /**
   * Attribute defines the type of filtering.
   */
  private final FilterMode filterMode;

  /**
   * Parameter can be used to include classes that match some specific pattern. The pattern matching mechanism is rather
   * simple. Thus only the following variants are supported:
   * <ul>
   * <li><code>com.anaptecs.jeaf.*</code> meaning that all classes whose name starts with "com.anaptecs.jeaf." will be
   * included</li>
   * <li><code>*.MessageConstant</code> meaning that all classes whose name ends with ".MessageConstant" will be
   * included</li>
   * <li><code>*.MessageConstant*</code> meaning that all classes whose name contains ".MessageConstant" will be
   * included</li>
   * <li><code>com.anaptecs.jeaf.components.MessageConstants</code> meaning that class with name
   * "com.anaptecs.jeaf.components.MessageConstants" will be included</li>
   * </ul>
   * 
   * Either configuration parameter "includes" or "excludes" can be used but not both.
   */
  private final List<String> filterExpressions;

  /**
   * Initialize object. Either <code>pIncludes</code> or <code>pExcludes</code> must be set or none but not both.
   * 
   * @param pIncludes List with include expressions. The parameter may be null.
   * @param pExcludes List with exclude expressions. The parameter may be null.
   */
  public NameFilter( List<String> pIncludes, List<String> pExcludes ) {
    // Defining includes and excludes together is not supported.
    if (pIncludes != null && pExcludes != null && pIncludes.isEmpty() == false && pExcludes.isEmpty() == false) {
      throw new RuntimeException(
          "Either configuration parameter 'includes' or 'excludes' or none can be used but not both.");
    }
    // Either includes or excludes or none is set.
    else {
      if (pIncludes != null && pIncludes.isEmpty() == false) {
        filterExpressions = new ArrayList<>(pIncludes);
        filterMode = FilterMode.INCLUDE;
      }
      else if (pExcludes != null && pExcludes.isEmpty() == false) {
        filterExpressions = new ArrayList<>(pExcludes);
        filterMode = FilterMode.EXCLUDE;
      }
      else {
        filterExpressions = Collections.emptyList();
        filterMode = FilterMode.NO_FILTERING;
      }
    }
  }

  /**
   * Method checks if the passed name is filtered out by the filter.
   * 
   * @param pName Name that should be checked.
   * @return boolean Method returns true if the passed name is filtered out and false otherwise.
   */
  public boolean isFilteredOut( String pName ) {
    // Check if passed name matches with any filter.
    boolean lMatchesFilter = false;
    for (String lNextFilter : filterExpressions) {
      lMatchesFilter = this.matchesFilter(pName, lNextFilter);
      if (lMatchesFilter == true) {
        break;
      }
    }

    // Depending on the filter mode a matching filter has a different meaning.
    boolean lFilteredOut;
    switch (filterMode) {
      // We are in exclude mode. This mean that if one of the filter expressions is applicable then the passed name will
      // be filtered out.
      case EXCLUDE:
        lFilteredOut = lMatchesFilter;
        break;

      // We are in include mode. This mean that if one of the filter expressions is applicable then the passed name will
      // not be filtered out.
      case INCLUDE:
        lFilteredOut = !lMatchesFilter;
        break;

      // No filtering at all
      case NO_FILTERING:
        lFilteredOut = false;
        break;

      // Unexpected filter mode.
      default:
        throw new RuntimeException("Unexpected filter mode: " + filterMode.name());
    }

    return lFilteredOut;
  }

  /**
   * Method checks if the passed name matches with the passed filter.
   * 
   * @param pName Name that should be checked for the filter. The parameter must not be null.
   * @param pFilter Filter expression that should be used. The parameter must not be null.
   * @return boolean Method returns true if the passed filter matches to the passed name and false otherwise.
   */
  private boolean matchesFilter( String pName, String pFilter ) {
    // Detect type of filter.
    FilterType lFilterType = this.detectFilterType(pFilter);

    // Check if class names matches with the passed filter.
    boolean lMatches;
    switch (lFilterType) {
      case STARTS_WITH:
        lMatches = pName.startsWith(pFilter.substring(0, pFilter.length() - 2));
        break;

      case ENDSWITH:
        lMatches = pName.endsWith(pFilter.substring(1));
        break;

      case CONTAINS:
        lMatches = pName.contains(pFilter.substring(1, pFilter.length() - 2));
        break;

      case EXACT_MATCH:
        lMatches = pName.equals(pFilter);
        break;

      default:
        throw new RuntimeException("Internal error: Unexpected filter type " + lFilterType.name());
    }
    return lMatches;
  }

  /**
   * Method detects the filter type depending on the passed filter expression.
   * 
   * @param pFilter Filter expression that should be analyzed. The parameter must not be null.
   * @return {@link FilterType} Detected filter type. The method never returns null.
   */
  private FilterType detectFilterType( String pFilter ) {
    FilterType lFilterType;
    if (pFilter.startsWith("*") && pFilter.endsWith("*")) {
      lFilterType = FilterType.CONTAINS;
    }
    else if (pFilter.startsWith("*")) {
      lFilterType = FilterType.ENDSWITH;
    }
    else if (pFilter.endsWith("*")) {
      lFilterType = FilterType.STARTS_WITH;
    }
    else {
      lFilterType = FilterType.EXACT_MATCH;
    }
    return lFilterType;
  }
}

enum FilterMode {
  INCLUDE, EXCLUDE, NO_FILTERING;
}

enum FilterType {
  STARTS_WITH, ENDSWITH, CONTAINS, EXACT_MATCH;
}
