/**
 * Copyright 2004 - 2019 anaptecs GmbH, Burgstr. 96, 72764 Reutlingen, Germany
 *
 * All rights reserved.
 */
package com.anaptecs.jeaf.maven;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Target;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.maven.plugin.logging.Log;

import com.anaptecs.jeaf.tools.api.Tools;
import com.anaptecs.jeaf.xfun.api.checks.Assert;

import io.github.classgraph.ClassInfo;
import io.github.classgraph.ClassInfoList;
import io.github.classgraph.ScanResult;

public class AnnotationBasedConfigGenerator {
  /**
   * Reference to logger for Maven Plugins.
   */
  private final Log log;

  /**
   * Configuration parameter defines the output directory to which the generated JEAF configuration files should be
   * written. The plugin will analyze therefore all dependencies that are not of scope "test".
   * 
   * At least one of "resourceGenDirsectory" or "testResourceGenDirectory" has to be set.
   */
  private final String resourceGenDirectory;

  /**
   * Configuration parameter defines the output directory to which the generated JEAF configuration files for test scope
   * should be written. The plugin will analyze therefore all dependencies including also those of scope "test".
   * 
   * At least one of "resourceGenDirectory" or "testResourceGenDirectory" has to be set.
   */
  private final String testResourceGenDirectory;

  /**
   * This mojo also supports writing file to target directory. There the target directory has to be passed through this
   * configuration parameter.
   */
  private final String targetDirectory;

  /**
   * This mojo also supports writing file to target test directory. There the target directory has to be passed through
   * this configuration parameter.
   */
  private final String testTargetDirectory;

  /**
   * As the scan result for the classpath is static we can pass it to this class when creating it.
   */
  private final ScanResult scanResult;

  private final List<String> classpath;

  public AnnotationBasedConfigGenerator( String pResourceGenDirectory, String pTestResourceGenDirectory,
      String pTargetDirectory, String pTestTargetDirectory, ScanResult pScanResult, Log pLog,
      List<String> pClasspath ) {

    // Check parameters
    Assert.assertNotNull(pScanResult, "pScanResult");
    Assert.assertNotNull(pClasspath, "pClasspath");
    Assert.assertNotNull(pLog, "pLog");

    resourceGenDirectory = pResourceGenDirectory;
    testResourceGenDirectory = pTestResourceGenDirectory;
    targetDirectory = pTargetDirectory;
    testTargetDirectory = pTestTargetDirectory;
    scanResult = pScanResult;
    classpath = pClasspath;
    log = pLog;
  }

  /**
   * Method generates configuration file for JEAF message resources.
   * 
   * @param pAnnotationProcessingConfig TODO
   * @param pTestScope If parameter is set to true then we work in test scope. This means that the file will be written
   * to the test output directory.
   * @param pScanResult Result of classpath scan. The parameter must not be null.
   * @throws IOException If an {@link IOException} occurs during writing the file.
   */
  public void generate( AnnotationProcessingConfig pAnnotationProcessingConfig, NameFilter pFilter, boolean pTestScope )
    throws IOException {

    // Check parameters
    Assert.assertNotNull(pAnnotationProcessingConfig.getAnnotation(), "AnnotationProcessingConfig.getAnnotation()");
    Assert.assertNotNull(pAnnotationProcessingConfig.getFilePath(), "AnnotationProcessingConfig.getFilePath()");
    Assert.assertNotNull(pFilter, "pFilter");

    // Analyze class path and generate output file.
    log.info(" ");
    List<String> lClassNames = this.detectAndFilterAnnotatedClasses(scanResult, pAnnotationProcessingConfig, pFilter);
    this.writeConfigFile(lClassNames, pAnnotationProcessingConfig.getFilePath(), pTestScope);
  }

  /**
   * Method detects all classes with the passed annotation and applies the passed filter on the found classes.
   * 
   * @param pScanResult Result of classpath scan. The parameter must not be null.
   * @param pAnnotation Annotation which should be detected inside the class path. The parameter must not be null.
   * @param pFilter Filter that should be applied. The parameter must not be null.
   * @param pPolicy Policy that should be used when finding more than one class with the defined annotation. The
   * parameter must not be null.
   * @return List with all class names that have the required annotation and are not filtered out. The method never
   * returns null.
   */
  private List<String> detectAndFilterAnnotatedClasses( ScanResult pScanResult, AnnotationProcessingConfig pConfig,
      NameFilter pFilter ) {

    // Check parameters
    Assert.assertNotNull(pScanResult, "pScanResult");
    Assert.assertNotNull(pConfig, "pConfig");
    Assert.assertNotNull(pFilter, "pFilter");

    // Resolve all classes with the passed annotation.
    Class<? extends Annotation> lAnnotation = pConfig.getAnnotation();
    ClassInfoList lAnnotatedClasses = this.detectAnnotatedClasses(pScanResult, lAnnotation);

    // Order classes according to classpath
    ClassInfoComparator lComparator = new ClassInfoComparator(classpath);
    lAnnotatedClasses.sort(lComparator);

    log.info("@" + lAnnotation.getSimpleName() + ":");

    // Process all annotations and collect class names and sort.
    List<String> lClassNamesList = new ArrayList<>();
    for (ClassInfo lNextClass : lAnnotatedClasses) {
      String lClassName = lNextClass.getName();

      // Check if class is filtered out due to the includes and excludes that were may be defined.
      if (pFilter.isFilteredOut(lClassName) == false) {
        // Class is not filtered out but we do not want to add it multiple times.
        if (lClassNamesList.contains(lClassName) == false) {
          lClassNamesList.add(lClassName);
          log.info("    " + lClassName + " (included)");
        }
      }
      // Ignoring class due to filter
      else {
        log.info("    " + lClassName + " (ignored due to defined includes / excludes)");
      }
    }
    // Apply class occurrence policy.
    ClassOccurrencePolicy lPolicy = pConfig.getPolicy();
    switch (lPolicy) {
      case ALL_CLASSES:
        // Nothing to do.
        break;

      case FIRST_CLASS:
        // Remove all elements from list except for the first one.
        int lSize = lClassNamesList.size();

        if (lSize > 0) {
          String lFirst = lClassNamesList.get(0);
          lClassNamesList.clear();
          lClassNamesList.add(lFirst);
          if (lSize > 1) {
            log.info("Using only annotated class " + lFirst
                + ". All other classes are ignored due to class occurrence configuiration "
                + ClassOccurrencePolicy.FIRST_CLASS.name() + ".");
          }
        }
        break;

      case ONE_CLASS_ONLY:
        if (lClassNamesList.size() > 1) {
          String lMessage = "Problem in classpath detected. We found more then one class with annotation @"
              + lAnnotation.getName()
              + ". Please cleanup your classpath or change configuration of occurrence policy of this Maven Plugin.";
          log.error(lMessage);
          throw new RuntimeException(lMessage);
        }
        break;

      case EXACTLY_ONE_CLASS_ONLY:
        if (lClassNamesList.size() > 1) {
          String lMessage = "Problem in classpath detected. We expect to find exactly one class with annotation @"
              + lAnnotation.getName()
              + ". Please cleanup your classpath or change configuration of occurrence policy of this Maven Plugin.";
          log.error(lMessage);
          throw new RuntimeException(lMessage);
        }
        break;
    }
    if (lClassNamesList.size() == 0) {
      log.info("No classes with annotation @" + lAnnotation.getSimpleName() + " found in classpath.");
    }
    log.info(" ");

    List<String> lClassNames = new ArrayList<>(lClassNamesList);

    // Class names will only be sorted in case that is requested.
    if (pConfig.isSorted()) {
      Collections.sort(lClassNames);
    }
    return lClassNames;
  }

  private ClassInfoList detectAnnotatedClasses( ScanResult pScanResult, Class<? extends Annotation> pAnnotation ) {
    String lAnnotationName = pAnnotation.getName();
    // Resolve the level on which the annotation is applied
    Target lTargetAnnotation = pAnnotation.getAnnotation(Target.class);
    List<ElementType> lElementTypes;
    if (lTargetAnnotation != null) {
      lElementTypes = Arrays.asList(lTargetAnnotation.value());
    }
    else {
      log.error(pAnnotation.getName() + " does not have a target annotation.");
      lElementTypes = new ArrayList<>();
      lElementTypes.add(ElementType.TYPE);
    }

    ClassInfoList lClassInfoList = new ClassInfoList();
    for (ElementType lNextType : lElementTypes) {
      switch (lNextType) {
        case TYPE:
          lClassInfoList.addAll(pScanResult.getClassesWithAnnotation(lAnnotationName));
          break;

        case METHOD:
          lClassInfoList.addAll(pScanResult.getClassesWithMethodAnnotation(lAnnotationName));
          break;

        case FIELD:
          lClassInfoList.addAll(pScanResult.getClassesWithFieldAnnotation(lAnnotationName));
          break;

        default:
          // Nothing to do.
      }
    }
    return lClassInfoList;
  }

  public void writeConfigFile( List<String> pLines, String pFilePath, boolean pTestScope ) throws IOException {
    // Create new file writer. A may be existing file will be overwritten
    Writer lWriter = null;
    try {
      File lOutputDirectory;
      File lOutputFile;
      File lOutputFileInTargetDir;

      // Handle directories for none test scope.
      if (pTestScope == false) {
        Assert.assertNotNull(resourceGenDirectory, "resourceGenDirectory");
        lOutputDirectory = new File(resourceGenDirectory + File.separatorChar + pFilePath).getParentFile();
        lOutputFile = new File(resourceGenDirectory + File.separatorChar + pFilePath);

        // We also support to add generated files also to target
        if (targetDirectory != null) {
          lOutputFileInTargetDir = new File(targetDirectory + File.separatorChar + pFilePath);
        }
        else {
          lOutputFileInTargetDir = null;
        }
      }
      // Handle directories for test scope.
      else {
        Assert.assertNotNull(testResourceGenDirectory, "testResourceGenDirectory");
        lOutputDirectory = new File(testResourceGenDirectory + File.separatorChar + pFilePath).getParentFile();
        lOutputFile = new File(testResourceGenDirectory + File.separatorChar + pFilePath);

        // We also support to add generated files also to target directory
        if (testTargetDirectory != null) {
          lOutputFileInTargetDir = new File(testTargetDirectory + File.separatorChar + pFilePath);
        }
        else {
          lOutputFileInTargetDir = null;
        }
      }
      // Create output directory if it does not exist
      if (lOutputDirectory.exists() == false) {
        lOutputDirectory.mkdirs();
      }

      // Delete previous version of file
      if (lOutputFile.exists()) {
        lOutputFile.delete();
      }
      // Create writer for standard output directory
      lWriter = new BufferedWriter(new FileWriter(lOutputFile, false));

      // Write config file
      for (String lNextLine : pLines) {
        lWriter.write(lNextLine);
        lWriter.write('\n');
      }
      lWriter.close();
      lWriter = null;

      if (pLines.isEmpty() == false) {
        log.info("Created JEAF config file " + lOutputFile.getCanonicalPath());
      }

      // Write file to target directory if requested.
      if (lOutputFileInTargetDir != null) {
        if (lOutputFileInTargetDir.exists()) {
          lOutputFileInTargetDir.delete();
        }
        else {
          File lParentDirectory = lOutputFileInTargetDir.getParentFile();
          if (lParentDirectory.exists() == false) {
            lParentDirectory.mkdirs();
          }
        }
        Tools.getFileTools().copyFile(lOutputFile, lOutputFileInTargetDir);
        if (pLines.isEmpty() == false) {
          log.info("Created configuration file also in target directory: " + lOutputFileInTargetDir.getCanonicalPath());
        }
      }
    }
    finally {
      if (lWriter != null) {
        lWriter.close();
      }
    }
  }
}
