package com.atlassian.bamboo.specs.api.model.pbc;

import com.atlassian.bamboo.specs.api.builders.AtlassianModule;
import com.atlassian.bamboo.specs.api.builders.pbc.ContainerSize;
import com.atlassian.bamboo.specs.api.model.AtlassianModuleProperties;
import com.atlassian.bamboo.specs.api.util.EntityPropertiesBuilders;
import com.atlassian.bamboo.specs.api.validators.common.ImporterUtils;
import com.atlassian.bamboo.specs.api.validators.common.ValidationContext;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;

class AbstractPerBuildContainerProperties {

    public static final ValidationContext VALIDATION_CONTEXT = ValidationContext.of("Per-Build Container (PBC)");

    private List<ExtraContainerProperties> extraContainers;
    private String size;
    private String image;
    private boolean enabled;
    protected final AtlassianModuleProperties module;
    
    
    protected AbstractPerBuildContainerProperties(AtlassianModule module) {
        this.module = EntityPropertiesBuilders.build(module);
    }
    

    public AbstractPerBuildContainerProperties(AtlassianModule module, boolean enabled, 
            String image, String size, List<ExtraContainerProperties> extraContainers) {
        this(module);
        this.enabled = enabled;
        this.image = image;
        this.size = size != null ? size.toUpperCase(Locale.ENGLISH) : null;
        this.extraContainers = extraContainers;
        validate();
    }
    
    public List<ExtraContainerProperties> getExtraContainers() {
        return extraContainers;
    }

    public String getSize() {
        return size;
    }

    public String getImage() {
        return image;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public final void validate() {
        if (enabled) {
            ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "image", image);
            ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                    image != null && StringUtils.deleteWhitespace(image).equals(image), 
                    "Argument 'image' cannot contain whitespace ('" + image + "').");
            ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "size", size);
            ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                    Stream.of(ContainerSize.values())
                            .anyMatch((ContainerSize t) -> {
                                return t.name().equals(size);
                            }),
                    "Container size is to be one of " + Arrays.toString(ContainerSize.values()));
            ImporterUtils.checkNotNull(VALIDATION_CONTEXT, "extraContainers", extraContainers);
            if (extraContainers != null) {
                extraContainers.forEach((ExtraContainerProperties t) -> {
                    t.validate();
                });
            }
        }
    }
    
    

    @Override
    public int hashCode() {
        return Objects.hash(enabled, image, size, extraContainers);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AbstractPerBuildContainerProperties other = (AbstractPerBuildContainerProperties) obj;
        if (this.enabled != other.enabled) {
            return false;
        }
        if (!Objects.equals(this.size, other.size)) {
            return false;
        }
        if (!Objects.equals(this.image, other.image)) {
            return false;
        }
        if (!Objects.equals(this.extraContainers, other.extraContainers)) {
            return false;
        }
        return true;
    }

 
    
    
}
