package com.atlassian.bamboo.specs.api.model.pbc;

import com.atlassian.bamboo.specs.api.builders.pbc.ExtraContainer;
import com.atlassian.bamboo.specs.api.builders.pbc.ExtraContainerSize;
import com.atlassian.bamboo.specs.api.codegen.annotations.Builder;
import com.atlassian.bamboo.specs.api.model.EntityProperties;
import com.atlassian.bamboo.specs.api.validators.common.ImporterUtils;
import com.atlassian.bamboo.specs.api.validators.common.ValidationContext;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;

@Builder(ExtraContainer.class)
public class ExtraContainerProperties implements EntityProperties {
    public static final ValidationContext VALIDATION_CONTEXT = 
            ValidationContext.of("Per-Build Container (PBC) Extra Container");

    private List<String> commands = new ArrayList<>();
    private String size;
    private List<EnvProperties> environments = new ArrayList<>();
    private String image;
    private String name;

    public ExtraContainerProperties() {
        
    }
    
    public ExtraContainerProperties(String name, String image, String size, 
            List<EnvProperties> environments, List<String> commands) {
        this.size = size != null ? size.toUpperCase(Locale.ENGLISH) : null;
        this.environments = environments;
        this.commands = commands;
        this.image = image;
        this.name = name;
    }

    public List<String> getCommands() {
        return commands;
    }

    public String getSize() {
        return size;
    }

    public List<EnvProperties> getEnvironments() {
        return environments;
    }

    public String getImage() {
        return image;
    }

    public String getName() {
        return name;
    }

    @Override
    public void validate() {
        ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "name", name);
        ImporterUtils.checkArgument(VALIDATION_CONTEXT, name != null && name.matches("[a-z0-9]([\\-a-z0-9]*[a-z0-9])?"),
                "Extra container name should be composed of lowercase letters, numbers and - character only");
        ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "image", image);
        ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                image != null && StringUtils.deleteWhitespace(image).equals(image), 
                "Argument 'image' cannot contain whitespace.('" + image + "').");
        ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "size", size);
        ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
            Stream.of(ExtraContainerSize.values())
                    .anyMatch((ExtraContainerSize t) -> {
                        return t.name().equals(size);
                    }),
            "Extra Container size is to be one of " + Arrays.toString(ExtraContainerSize.values()));

        ImporterUtils.checkNotNull(VALIDATION_CONTEXT, "commands", commands);
        if (commands != null) {
            commands.forEach((String t) -> {
                ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "command '" + t + "'", t);
            });
        }
        ImporterUtils.checkNotNull(VALIDATION_CONTEXT, "envVariables", environments);
        if (environments != null) {
            environments.forEach((EnvProperties t) -> {
                t.validate();
            });
        }
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, image, environments, size, commands);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ExtraContainerProperties other = (ExtraContainerProperties) obj;
        if (!Objects.equals(this.size, other.size)) {
            return false;
        }
        if (!Objects.equals(this.image, other.image)) {
            return false;
        }
        if (!Objects.equals(this.name, other.name)) {
            return false;
        }
        if (!Objects.equals(this.commands, other.commands)) {
            return false;
        }
        if (!Objects.equals(this.environments, other.environments)) {
            return false;
        }
        return true;
    }

}
