package com.atlassian.bamboo.specs.api.builders.pbc;

import com.atlassian.bamboo.specs.api.builders.plan.configuration.PluginConfiguration;
import com.atlassian.bamboo.specs.api.model.pbc.ExtraContainerProperties;
import com.atlassian.bamboo.specs.api.model.pbc.PerBuildContainerForJobProperties;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Per Build Container definition for Bamboo Jobs.
 */
public class PerBuildContainerForJob extends PluginConfiguration<PerBuildContainerForJobProperties> {


    private String image;
    private boolean enabled = true;
    private String size = ContainerSize.REGULAR.name();
    private String awsRole;
    private String architecture;
    private List<ExtraContainerProperties> extraContainers = new ArrayList<>();

    public PerBuildContainerForJob() {
    }
    
    /**
     * Marks PBC as enabled for given job. Default value is true.
     */
    public PerBuildContainerForJob enabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }
    
    public PerBuildContainerForJob image(String image) {
        this.image = image;
        return this;
    }

    /**
     * Symbolic name for size of the container.
     * See ContainerSize enum for allowed values.
     * The default value is 'REGULAR'.
     */
    public PerBuildContainerForJob size(String size) {
        this.size = size;
        return this;
    }
    
    /**
     * Symbolic name for size of the container.
     * The default value is 'REGULAR'.
     */
    public PerBuildContainerForJob size(ContainerSize size) {
        this.size = size != null ? size.name() : null;
        return this;
    }
    
    /**
     * AWS Role of the current build.
     * Optional. Needs to be supported by the runtime environment.
     */
    public PerBuildContainerForJob awsRole(String awsRole) {
        this.awsRole = awsRole;
        return this;
    }

    /**
     * Architecture the build should run on
     * Optional. If specified, the Bamboo server must have this configured as one of the available architectures.
     */
    public PerBuildContainerForJob architecture(String architecture) {
        this.architecture = architecture;
        return this;
    }
    
    public PerBuildContainerForJob extraContainers(ExtraContainer... extraContainers) {
        extraContainers(Arrays.asList(extraContainers));
        return this;
    }
    
    public PerBuildContainerForJob extraContainers(List<ExtraContainer> extraContainers) {
        extraContainers.stream().map(ExtraContainer::build)
                .forEach(this.extraContainers::add);
        return this;
    }
    
    
    @Override
    protected PerBuildContainerForJobProperties build() {
        return new PerBuildContainerForJobProperties(enabled, image, size, extraContainers, awsRole, architecture);
    }

    
}
