package com.atlassian.bamboo.specs.api.model.pbc;

import com.atlassian.bamboo.specs.api.builders.AtlassianModule;
import com.atlassian.bamboo.specs.api.builders.pbc.ContainerSize;
import com.atlassian.bamboo.specs.api.model.AtlassianModuleProperties;
import com.atlassian.bamboo.specs.api.util.EntityPropertiesBuilders;
import com.atlassian.bamboo.specs.api.validators.common.ImporterUtils;
import com.atlassian.bamboo.specs.api.validators.common.ValidationContext;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;

class AbstractPerBuildContainerProperties {

    public static final ValidationContext VALIDATION_CONTEXT = ValidationContext.of("Per-Build Container (PBC)");

    private List<ExtraContainerProperties> extraContainers;
    private String size;
    private String image;
    private boolean enabled;
    protected final AtlassianModuleProperties module;
    private String awsRole;
    
    
    protected AbstractPerBuildContainerProperties(AtlassianModule module) {
        this.module = EntityPropertiesBuilders.build(module);
    }
    

    public AbstractPerBuildContainerProperties(AtlassianModule module, boolean enabled, 
            String image, String size, List<ExtraContainerProperties> extraContainers, String awsRole) {
        this(module);
        this.enabled = enabled;
        this.image = image;
        this.size = size != null ? size.toUpperCase(Locale.ENGLISH) : null;
        this.extraContainers = extraContainers;
        this.awsRole = awsRole;
        validate();
    }
    
    public List<ExtraContainerProperties> getExtraContainers() {
        return extraContainers;
    }

    public String getSize() {
        return size;
    }

    public String getImage() {
        return image;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public String getAwsRole() {
        return awsRole;
    }

    public final void validate() {
        if (enabled) {
            ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "image", image);
            ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                    image != null && StringUtils.deleteWhitespace(image).equals(image), 
                    "Argument 'image' cannot contain whitespace ('" + image + "').");
            ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "size", size);
            ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                    Stream.of(ContainerSize.values())
                            .anyMatch((ContainerSize t) -> {
                                return t.name().equals(size);
                            }),
                    "Container size is to be one of " + Arrays.toString(ContainerSize.values()));
            ImporterUtils.checkNotNull(VALIDATION_CONTEXT, "extraContainers", extraContainers);
            if (extraContainers != null) {
                extraContainers.forEach((ExtraContainerProperties t) -> {
                    t.validate();
                });
            }
            if (awsRole != null) {
                ImporterUtils.checkNotBlank(VALIDATION_CONTEXT, "awsRole", awsRole);
                ImporterUtils.checkArgument(VALIDATION_CONTEXT, 
                    Pattern.compile("arn:aws:iam::[0-9]+:role/[a-zA-Z0-9+=,.@_\\-]+").matcher(awsRole).matches(),
                    "Format of IAM role ARN is wrong.");
            }
        }
    }

    

    @Override
    public int hashCode() {
        return Objects.hash(enabled, image, size, extraContainers, awsRole);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AbstractPerBuildContainerProperties other = (AbstractPerBuildContainerProperties) obj;
        if (this.enabled != other.enabled) {
            return false;
        }
        if (!Objects.equals(this.size, other.size)) {
            return false;
        }
        if (!Objects.equals(this.image, other.image)) {
            return false;
        }
        if (!Objects.equals(this.awsRole, other.awsRole)) {
            return false;
        }
        if (!Objects.equals(this.extraContainers, other.extraContainers)) {
            return false;
        }
        return true;
    }

}
