;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.budgets.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.budgets/EventType
  (s/spec
    string?
    :gen
    #(s/gen #{"SYSTEM" "DELETE_ACTION" "UPDATE_ACTION" "CREATE_ACTION" "EXECUTE_ACTION"})))

(s/def
  :cognitect.aws.budgets/ActionHistory
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ActionHistory/Timestamp
     :cognitect.aws.budgets.ActionHistory/Status
     :cognitect.aws.budgets.ActionHistory/EventType
     :cognitect.aws.budgets.ActionHistory/ActionHistoryDetails]))

(s/def
  :cognitect.aws.budgets/DuplicateRecordException
  (s/keys :opt-un [:cognitect.aws.budgets.DuplicateRecordException/Message]))

(s/def
  :cognitect.aws.budgets/ThrottlingException
  (s/keys :opt-un [:cognitect.aws.budgets.ThrottlingException/Message]))

(s/def
  :cognitect.aws.budgets/CreateBudgetActionRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CreateBudgetActionRequest/AccountId
     :cognitect.aws.budgets.CreateBudgetActionRequest/BudgetName
     :cognitect.aws.budgets.CreateBudgetActionRequest/NotificationType
     :cognitect.aws.budgets.CreateBudgetActionRequest/ActionType
     :cognitect.aws.budgets.CreateBudgetActionRequest/ActionThreshold
     :cognitect.aws.budgets.CreateBudgetActionRequest/Definition
     :cognitect.aws.budgets.CreateBudgetActionRequest/ExecutionRoleArn
     :cognitect.aws.budgets.CreateBudgetActionRequest/ApprovalModel
     :cognitect.aws.budgets.CreateBudgetActionRequest/Subscribers]
    :opt-un
    [:cognitect.aws.budgets.CreateBudgetActionRequest/ResourceTags]))

(s/def :cognitect.aws.budgets/ApprovalModel (s/spec string? :gen #(s/gen #{"MANUAL" "AUTOMATIC"})))

(s/def
  :cognitect.aws.budgets/UnitValue
  (s/spec #(re-matches (re-pattern ".*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.TagResourceRequest/ResourceARN
     :cognitect.aws.budgets.TagResourceRequest/ResourceTags]))

(s/def
  :cognitect.aws.budgets/ServiceQuotaExceededException
  (s/keys :opt-un [:cognitect.aws.budgets.ServiceQuotaExceededException/Message]))

(s/def :cognitect.aws.budgets/Expressions (s/coll-of :cognitect.aws.budgets/Expression))

(s/def
  :cognitect.aws.budgets/TargetId
  (s/spec
    #(re-matches (re-pattern "^(ou-[0-9a-z]{4,32}-[a-z0-9]{8,32}$)|(\\d{12})") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/Group
  (s/spec
    #(re-matches (re-pattern "^([\\u0021-\\u007F]+\\u002F)?[\\w+=,.@-]+$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/ActionHistories
  (s/coll-of :cognitect.aws.budgets/ActionHistory :min-count 0 :max-count 100))

(s/def :cognitect.aws.budgets/DeleteSubscriberResponse (s/keys))

(s/def
  :cognitect.aws.budgets/NumericValue
  (s/spec #(re-matches (re-pattern "([0-9]*\\.)?[0-9]+") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/Actions
  (s/coll-of :cognitect.aws.budgets/Action :min-count 0 :max-count 100))

(s/def
  :cognitect.aws.budgets/ActionThreshold
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ActionThreshold/ActionThresholdValue
     :cognitect.aws.budgets.ActionThreshold/ActionThresholdType]))

(s/def :cognitect.aws.budgets/DeleteNotificationResponse (s/keys))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionsForAccountRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/AccountId]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/CreateBudgetActionResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CreateBudgetActionResponse/AccountId
     :cognitect.aws.budgets.CreateBudgetActionResponse/BudgetName
     :cognitect.aws.budgets.CreateBudgetActionResponse/ActionId]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionsForBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/AccountId
     :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/BudgetName]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/Groups
  (s/coll-of :cognitect.aws.budgets/Group :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.budgets/Dimension
  (s/spec
    string?
    :gen
    #(s/gen
      #{"SUBSCRIPTION_ID"
        "SERVICE_CODE"
        "LINKED_ACCOUNT"
        "LEGAL_ENTITY_NAME"
        "TENANCY"
        "PURCHASE_TYPE"
        "PAYMENT_OPTION"
        "SERVICE"
        "SAVINGS_PLANS_TYPE"
        "TAG_KEY"
        "INSTANCE_TYPE_FAMILY"
        "RECORD_TYPE"
        "SAVINGS_PLAN_ARN"
        "INSTANCE_TYPE"
        "DEPLOYMENT_OPTION"
        "RESERVATION_MODIFIED"
        "LINKED_ACCOUNT_NAME"
        "COST_CATEGORY_NAME"
        "PLATFORM"
        "OPERATING_SYSTEM"
        "RESOURCE_ID"
        "SCOPE"
        "CACHE_ENGINE"
        "USAGE_TYPE_GROUP"
        "DATABASE_ENGINE"
        "REGION"
        "RIGHTSIZING_TYPE"
        "AZ"
        "RESERVATION_ID"
        "INVOICING_ENTITY"
        "BILLING_ENTITY"
        "OPERATION"
        "USAGE_TYPE"})))

(s/def :cognitect.aws.budgets/UpdateBudgetResponse (s/keys))

(s/def :cognitect.aws.budgets/TagResourceResponse (s/keys))

(s/def
  :cognitect.aws.budgets/ExecuteBudgetActionRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ExecuteBudgetActionRequest/AccountId
     :cognitect.aws.budgets.ExecuteBudgetActionRequest/BudgetName
     :cognitect.aws.budgets.ExecuteBudgetActionRequest/ActionId
     :cognitect.aws.budgets.ExecuteBudgetActionRequest/ExecutionType]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionRequest/AccountId
     :cognitect.aws.budgets.DescribeBudgetActionRequest/BudgetName
     :cognitect.aws.budgets.DescribeBudgetActionRequest/ActionId]))

(s/def :cognitect.aws.budgets/CostCategoryName string?)

(s/def :cognitect.aws.budgets/errorMessage string?)

(s/def :cognitect.aws.budgets/CreateBudgetResponse (s/keys))

(s/def
  :cognitect.aws.budgets/AutoAdjustType
  (s/spec string? :gen #(s/gen #{"FORECAST" "HISTORICAL"})))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionResponse/AccountId
     :cognitect.aws.budgets.DescribeBudgetActionResponse/BudgetName
     :cognitect.aws.budgets.DescribeBudgetActionResponse/Action]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionsForBudgetResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForBudgetResponse/Actions]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForBudgetResponse/NextToken]))

(s/def
  :cognitect.aws.budgets/DeleteBudgetActionResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DeleteBudgetActionResponse/AccountId
     :cognitect.aws.budgets.DeleteBudgetActionResponse/BudgetName
     :cognitect.aws.budgets.DeleteBudgetActionResponse/Action]))

(s/def
  :cognitect.aws.budgets/ResourceTagList
  (s/coll-of :cognitect.aws.budgets/ResourceTag :min-count 0 :max-count 200))

(s/def
  :cognitect.aws.budgets/SubscriberAddress
  (s/spec #(re-matches (re-pattern "(.*[\\n\\r\\t\\f\\ ]?)*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/DescribeSubscribersForNotificationRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/AccountId
     :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/BudgetName
     :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/Notification]
    :opt-un
    [:cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/NextToken
     :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/AccessDeniedException
  (s/keys :opt-un [:cognitect.aws.budgets.AccessDeniedException/Message]))

(s/def
  :cognitect.aws.budgets/Notification
  (s/keys
    :req-un
    [:cognitect.aws.budgets.Notification/NotificationType
     :cognitect.aws.budgets.Notification/ComparisonOperator
     :cognitect.aws.budgets.Notification/Threshold]
    :opt-un
    [:cognitect.aws.budgets.Notification/ThresholdType
     :cognitect.aws.budgets.Notification/NotificationState]))

(s/def
  :cognitect.aws.budgets/CreateBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CreateBudgetRequest/AccountId
     :cognitect.aws.budgets.CreateBudgetRequest/Budget]
    :opt-un
    [:cognitect.aws.budgets.CreateBudgetRequest/ResourceTags
     :cognitect.aws.budgets.CreateBudgetRequest/NotificationsWithSubscribers]))

(s/def
  :cognitect.aws.budgets/CalculatedSpend
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CalculatedSpend/ActualSpend]
    :opt-un
    [:cognitect.aws.budgets.CalculatedSpend/ForecastedSpend]))

(s/def
  :cognitect.aws.budgets/AccountId
  (s/spec #(re-matches (re-pattern "\\d{12}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/CostFilters
  (s/map-of :cognitect.aws.budgets/GenericString :cognitect.aws.budgets/DimensionValues))

(s/def
  :cognitect.aws.budgets/ResourceTag
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ResourceTag/Key :cognitect.aws.budgets.ResourceTag/Value]))

(s/def
  :cognitect.aws.budgets/TagValues
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.TagValues/Key
     :cognitect.aws.budgets.TagValues/Values
     :cognitect.aws.budgets.TagValues/MatchOptions]))

(s/def :cognitect.aws.budgets/CreateSubscriberResponse (s/keys))

(s/def
  :cognitect.aws.budgets/PolicyArn
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:aws(-cn|-us-gov|-iso|-iso-[a-z]{1})?:iam::(\\d{12}|aws):policy(\\u002F[\\u0021-\\u007F]+\\u002F|\\u002F)[\\w+=,.@-]+$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/ActionHistoryDetails
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ActionHistoryDetails/Message
     :cognitect.aws.budgets.ActionHistoryDetails/Action]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetNotificationsForAccountRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/AccountId]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/CostCategoryValues
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.CostCategoryValues/Key
     :cognitect.aws.budgets.CostCategoryValues/Values
     :cognitect.aws.budgets.CostCategoryValues/MatchOptions]))

(s/def
  :cognitect.aws.budgets/InstanceId
  (s/spec
    #(re-matches (re-pattern "^i-(\\w{8}|\\w{17})$|^[a-zA-Z]([\\w-]{0,61}\\w)?$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/ResourceTagKey
  (s/spec
    (s/and string? #(<= 1 (count %) 128))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 128) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.budgets/ActionType
  (s/spec string? :gen #(s/gen #{"APPLY_SCP_POLICY" "RUN_SSM_DOCUMENTS" "APPLY_IAM_POLICY"})))

(s/def
  :cognitect.aws.budgets/Budget
  (s/keys
    :req-un
    [:cognitect.aws.budgets.Budget/BudgetName
     :cognitect.aws.budgets.Budget/TimeUnit
     :cognitect.aws.budgets.Budget/BudgetType]
    :opt-un
    [:cognitect.aws.budgets.Budget/CalculatedSpend
     :cognitect.aws.budgets.Budget/BudgetLimit
     :cognitect.aws.budgets.Budget/CostTypes
     :cognitect.aws.budgets.Budget/TimePeriod
     :cognitect.aws.budgets.Budget/FilterExpression
     :cognitect.aws.budgets.Budget/PlannedBudgetLimits
     :cognitect.aws.budgets.Budget/LastUpdatedTime
     :cognitect.aws.budgets.Budget/AutoAdjustData
     :cognitect.aws.budgets.Budget/CostFilters
     :cognitect.aws.budgets.Budget/Metrics]))

(s/def
  :cognitect.aws.budgets/ExecuteBudgetActionResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ExecuteBudgetActionResponse/AccountId
     :cognitect.aws.budgets.ExecuteBudgetActionResponse/BudgetName
     :cognitect.aws.budgets.ExecuteBudgetActionResponse/ActionId
     :cognitect.aws.budgets.ExecuteBudgetActionResponse/ExecutionType]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetNotificationsForAccountResponse
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetNotificationsForAccountResponse/NextToken
     :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountResponse/BudgetNotificationsForAccount]))

(s/def
  :cognitect.aws.budgets/ResourceLockedException
  (s/keys :opt-un [:cognitect.aws.budgets.ResourceLockedException/Message]))

(s/def
  :cognitect.aws.budgets/AutoAdjustData
  (s/keys
    :req-un
    [:cognitect.aws.budgets.AutoAdjustData/AutoAdjustType]
    :opt-un
    [:cognitect.aws.budgets.AutoAdjustData/HistoricalOptions
     :cognitect.aws.budgets.AutoAdjustData/LastAutoAdjustTime]))

(s/def
  :cognitect.aws.budgets/NotFoundException
  (s/keys :opt-un [:cognitect.aws.budgets.NotFoundException/Message]))

(s/def
  :cognitect.aws.budgets/UpdateNotificationRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UpdateNotificationRequest/AccountId
     :cognitect.aws.budgets.UpdateNotificationRequest/BudgetName
     :cognitect.aws.budgets.UpdateNotificationRequest/OldNotification
     :cognitect.aws.budgets.UpdateNotificationRequest/NewNotification]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetsRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetsRequest/AccountId]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetsRequest/ShowFilterExpression
     :cognitect.aws.budgets.DescribeBudgetsRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetsRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/UpdateBudgetActionResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UpdateBudgetActionResponse/AccountId
     :cognitect.aws.budgets.UpdateBudgetActionResponse/BudgetName
     :cognitect.aws.budgets.UpdateBudgetActionResponse/OldAction
     :cognitect.aws.budgets.UpdateBudgetActionResponse/NewAction]))

(s/def
  :cognitect.aws.budgets/TargetIds
  (s/coll-of :cognitect.aws.budgets/TargetId :min-count 1 :max-count 100))

(s/def :cognitect.aws.budgets/DimensionValues (s/coll-of :cognitect.aws.budgets/DimensionValue))

(s/def
  :cognitect.aws.budgets/UpdateBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UpdateBudgetRequest/AccountId
     :cognitect.aws.budgets.UpdateBudgetRequest/NewBudget]))

(s/def
  :cognitect.aws.budgets/ComparisonOperator
  (s/spec string? :gen #(s/gen #{"GREATER_THAN" "LESS_THAN" "EQUAL_TO"})))

(s/def
  :cognitect.aws.budgets/ResourceTagValue
  (s/spec
    (s/and string? #(<= 0 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.budgets/NullableBoolean boolean?)

(s/def
  :cognitect.aws.budgets/AmazonResourceName
  (s/spec
    (s/and string? #(<= 1 (count %) 1011))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1011) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.budgets/CreateNotificationResponse (s/keys))

(s/def
  :cognitect.aws.budgets/ActionId
  (s/spec
    #(re-matches
      (re-pattern "^[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/HistoricalOptions
  (s/keys
    :req-un
    [:cognitect.aws.budgets.HistoricalOptions/BudgetAdjustmentPeriod]
    :opt-un
    [:cognitect.aws.budgets.HistoricalOptions/LookBackAvailablePeriods]))

(s/def :cognitect.aws.budgets/Metrics (s/coll-of :cognitect.aws.budgets/Metric :max-count 1))

(s/def
  :cognitect.aws.budgets/BudgetPerformanceHistory
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.BudgetPerformanceHistory/TimeUnit
     :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetedAndActualAmountsList
     :cognitect.aws.budgets.BudgetPerformanceHistory/CostTypes
     :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetName
     :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetType
     :cognitect.aws.budgets.BudgetPerformanceHistory/CostFilters]))

(s/def
  :cognitect.aws.budgets/DeleteSubscriberRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DeleteSubscriberRequest/AccountId
     :cognitect.aws.budgets.DeleteSubscriberRequest/BudgetName
     :cognitect.aws.budgets.DeleteSubscriberRequest/Notification
     :cognitect.aws.budgets.DeleteSubscriberRequest/Subscriber]))

(s/def
  :cognitect.aws.budgets/Definition
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.Definition/ScpActionDefinition
     :cognitect.aws.budgets.Definition/SsmActionDefinition
     :cognitect.aws.budgets.Definition/IamActionDefinition]))

(s/def
  :cognitect.aws.budgets/BudgetNotificationsForAccount
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.BudgetNotificationsForAccount/BudgetName
     :cognitect.aws.budgets.BudgetNotificationsForAccount/Notifications]))

(s/def
  :cognitect.aws.budgets/AdjustmentPeriod
  (s/spec (s/and int? #(<= 1 % 60)) :gen #(gen/choose 1 60)))

(s/def
  :cognitect.aws.budgets/Expression
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.Expression/And
     :cognitect.aws.budgets.Expression/CostCategories
     :cognitect.aws.budgets.Expression/Tags
     :cognitect.aws.budgets.Expression/Not
     :cognitect.aws.budgets.Expression/Dimensions
     :cognitect.aws.budgets.Expression/Or]))

(s/def
  :cognitect.aws.budgets/ExpiredNextTokenException
  (s/keys :opt-un [:cognitect.aws.budgets.ExpiredNextTokenException/Message]))

(s/def
  :cognitect.aws.budgets/DeleteBudgetActionRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DeleteBudgetActionRequest/AccountId
     :cognitect.aws.budgets.DeleteBudgetActionRequest/BudgetName
     :cognitect.aws.budgets.DeleteBudgetActionRequest/ActionId]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionHistoriesRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/AccountId
     :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/BudgetName
     :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/ActionId]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/TimePeriod
     :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/CreationLimitExceededException
  (s/keys :opt-un [:cognitect.aws.budgets.CreationLimitExceededException/Message]))

(s/def
  :cognitect.aws.budgets/MatchOption
  (s/spec
    string?
    :gen
    #(s/gen
      #{"GREATER_THAN_OR_EQUAL"
        "ABSENT"
        "CONTAINS"
        "CASE_SENSITIVE"
        "CASE_INSENSITIVE"
        "ENDS_WITH"
        "EQUALS"
        "STARTS_WITH"})))

(s/def
  :cognitect.aws.budgets/ResourceTagKeyList
  (s/coll-of :cognitect.aws.budgets/ResourceTagKey :min-count 0 :max-count 200))

(s/def
  :cognitect.aws.budgets/BudgetedAndActualAmountsList
  (s/coll-of :cognitect.aws.budgets/BudgetedAndActualAmounts))

(s/def
  :cognitect.aws.budgets/CreateNotificationRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CreateNotificationRequest/AccountId
     :cognitect.aws.budgets.CreateNotificationRequest/BudgetName
     :cognitect.aws.budgets.CreateNotificationRequest/Notification
     :cognitect.aws.budgets.CreateNotificationRequest/Subscribers]))

(s/def
  :cognitect.aws.budgets/ListTagsForResourceRequest
  (s/keys :req-un [:cognitect.aws.budgets.ListTagsForResourceRequest/ResourceARN]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetRequest/AccountId
     :cognitect.aws.budgets.DescribeBudgetRequest/BudgetName]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetRequest/ShowFilterExpression]))

(s/def
  :cognitect.aws.budgets/NotificationThreshold
  (s/spec
    (s/and double? #(<= 0 % 15000000000000))
    :gen
    #(gen/double* {:infinite? false, :NaN? false, :min 0, :max 15000000000000})))

(s/def :cognitect.aws.budgets/Values (s/coll-of :cognitect.aws.budgets/Value :min-count 1))

(s/def :cognitect.aws.budgets/SubscriptionType (s/spec string? :gen #(s/gen #{"SNS" "EMAIL"})))

(s/def
  :cognitect.aws.budgets/InvalidParameterException
  (s/keys :opt-un [:cognitect.aws.budgets.InvalidParameterException/Message]))

(s/def
  :cognitect.aws.budgets/RoleArn
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:aws(-cn|-us-gov|-iso|-iso-[a-z]{1})?:iam::\\d{12}:role(\\u002F[\\u0021-\\u007F]+\\u002F|\\u002F)[\\w+=,.@-]+$")
      %)
    :gen
    #(gen/string)))

(s/def :cognitect.aws.budgets/UntagResourceResponse (s/keys))

(s/def :cognitect.aws.budgets/DeleteBudgetResponse (s/keys))

(s/def
  :cognitect.aws.budgets/User
  (s/spec
    #(re-matches (re-pattern "^([\\u0021-\\u007F]+\\u002F)?[\\w+=,.@-]+$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/DescribeNotificationsForBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/AccountId
     :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/BudgetName]
    :opt-un
    [:cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/NextToken
     :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/Region
  (s/spec #(re-matches (re-pattern "^\\w{2}-\\w+(-\\w+)?-\\d$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/UpdateBudgetActionRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UpdateBudgetActionRequest/AccountId
     :cognitect.aws.budgets.UpdateBudgetActionRequest/BudgetName
     :cognitect.aws.budgets.UpdateBudgetActionRequest/ActionId]
    :opt-un
    [:cognitect.aws.budgets.UpdateBudgetActionRequest/ActionThreshold
     :cognitect.aws.budgets.UpdateBudgetActionRequest/ApprovalModel
     :cognitect.aws.budgets.UpdateBudgetActionRequest/NotificationType
     :cognitect.aws.budgets.UpdateBudgetActionRequest/ExecutionRoleArn
     :cognitect.aws.budgets.UpdateBudgetActionRequest/Subscribers
     :cognitect.aws.budgets.UpdateBudgetActionRequest/Definition]))

(s/def
  :cognitect.aws.budgets/Spend
  (s/keys :req-un [:cognitect.aws.budgets.Spend/Amount :cognitect.aws.budgets.Spend/Unit]))

(s/def :cognitect.aws.budgets/UpdateSubscriberResponse (s/keys))

(s/def
  :cognitect.aws.budgets/IamActionDefinition
  (s/keys
    :req-un
    [:cognitect.aws.budgets.IamActionDefinition/PolicyArn]
    :opt-un
    [:cognitect.aws.budgets.IamActionDefinition/Roles
     :cognitect.aws.budgets.IamActionDefinition/Users
     :cognitect.aws.budgets.IamActionDefinition/Groups]))

(s/def
  :cognitect.aws.budgets/SsmActionDefinition
  (s/keys
    :req-un
    [:cognitect.aws.budgets.SsmActionDefinition/ActionSubType
     :cognitect.aws.budgets.SsmActionDefinition/Region
     :cognitect.aws.budgets.SsmActionDefinition/InstanceIds]))

(s/def
  :cognitect.aws.budgets/MaxResultsBudgetNotifications
  (s/spec (s/and int? #(<= 1 % 1000)) :gen #(gen/choose 1 1000)))

(s/def
  :cognitect.aws.budgets/ExecutionType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"REVERSE_BUDGET_ACTION"
        "RESET_BUDGET_ACTION"
        "APPROVE_BUDGET_ACTION"
        "RETRY_BUDGET_ACTION"})))

(s/def :cognitect.aws.budgets/MatchOptions (s/coll-of :cognitect.aws.budgets/MatchOption))

(s/def
  :cognitect.aws.budgets/DescribeBudgetPerformanceHistoryRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/AccountId
     :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/BudgetName]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/NextToken
     :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/TimePeriod
     :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/MaxResults]))

(s/def
  :cognitect.aws.budgets/DeleteBudgetRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DeleteBudgetRequest/AccountId
     :cognitect.aws.budgets.DeleteBudgetRequest/BudgetName]))

(s/def
  :cognitect.aws.budgets/Users
  (s/coll-of :cognitect.aws.budgets/User :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.budgets/InstanceIds
  (s/coll-of :cognitect.aws.budgets/InstanceId :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.budgets/InternalErrorException
  (s/keys :opt-un [:cognitect.aws.budgets.InternalErrorException/Message]))

(s/def
  :cognitect.aws.budgets/TimePeriod
  (s/keys :opt-un [:cognitect.aws.budgets.TimePeriod/Start :cognitect.aws.budgets.TimePeriod/End]))

(s/def
  :cognitect.aws.budgets/ActionStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"REVERSE_IN_PROGRESS"
        "REVERSE_FAILURE"
        "REVERSE_SUCCESS"
        "PENDING"
        "RESET_IN_PROGRESS"
        "STANDBY"
        "EXECUTION_SUCCESS"
        "EXECUTION_FAILURE"
        "EXECUTION_IN_PROGRESS"
        "RESET_FAILURE"})))

(s/def
  :cognitect.aws.budgets/TimeUnit
  (s/spec string? :gen #(s/gen #{"QUARTERLY" "MONTHLY" "DAILY" "ANNUALLY"})))

(s/def
  :cognitect.aws.budgets/Action
  (s/keys
    :req-un
    [:cognitect.aws.budgets.Action/ActionId
     :cognitect.aws.budgets.Action/BudgetName
     :cognitect.aws.budgets.Action/NotificationType
     :cognitect.aws.budgets.Action/ActionType
     :cognitect.aws.budgets.Action/ActionThreshold
     :cognitect.aws.budgets.Action/Definition
     :cognitect.aws.budgets.Action/ExecutionRoleArn
     :cognitect.aws.budgets.Action/ApprovalModel
     :cognitect.aws.budgets.Action/Status
     :cognitect.aws.budgets.Action/Subscribers]))

(s/def
  :cognitect.aws.budgets/BudgetedAndActualAmounts
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.BudgetedAndActualAmounts/TimePeriod
     :cognitect.aws.budgets.BudgetedAndActualAmounts/ActualAmount
     :cognitect.aws.budgets.BudgetedAndActualAmounts/BudgetedAmount]))

(s/def :cognitect.aws.budgets/GenericTimestamp inst?)

(s/def
  :cognitect.aws.budgets/BudgetNotificationsForAccountList
  (s/coll-of :cognitect.aws.budgets/BudgetNotificationsForAccount :max-count 1000))

(s/def
  :cognitect.aws.budgets/GenericString
  (s/spec #(re-matches (re-pattern ".*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/MaxResultsDescribeBudgets
  (s/spec (s/and int? #(<= 1 % 1000)) :gen #(gen/choose 1 1000)))

(s/def
  :cognitect.aws.budgets/CostTypes
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.CostTypes/IncludeSupport
     :cognitect.aws.budgets.CostTypes/IncludeOtherSubscription
     :cognitect.aws.budgets.CostTypes/IncludeSubscription
     :cognitect.aws.budgets.CostTypes/IncludeUpfront
     :cognitect.aws.budgets.CostTypes/UseBlended
     :cognitect.aws.budgets.CostTypes/IncludeTax
     :cognitect.aws.budgets.CostTypes/IncludeCredit
     :cognitect.aws.budgets.CostTypes/IncludeDiscount
     :cognitect.aws.budgets.CostTypes/IncludeRefund
     :cognitect.aws.budgets.CostTypes/IncludeRecurring
     :cognitect.aws.budgets.CostTypes/UseAmortized]))

(s/def
  :cognitect.aws.budgets/Value
  (s/spec #(re-matches (re-pattern "[\\S\\s]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/Subscriber
  (s/keys
    :req-un
    [:cognitect.aws.budgets.Subscriber/SubscriptionType :cognitect.aws.budgets.Subscriber/Address]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetResponse
  (s/keys :opt-un [:cognitect.aws.budgets.DescribeBudgetResponse/Budget]))

(s/def
  :cognitect.aws.budgets/MaxResults
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def
  :cognitect.aws.budgets/ActionSubType
  (s/spec string? :gen #(s/gen #{"STOP_RDS_INSTANCES" "STOP_EC2_INSTANCES"})))

(s/def
  :cognitect.aws.budgets/DescribeBudgetPerformanceHistoryResponse
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetPerformanceHistoryResponse/NextToken
     :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryResponse/BudgetPerformanceHistory]))

(s/def
  :cognitect.aws.budgets/Roles
  (s/coll-of :cognitect.aws.budgets/Role :min-count 1 :max-count 100))

(s/def
  :cognitect.aws.budgets/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UntagResourceRequest/ResourceARN
     :cognitect.aws.budgets.UntagResourceRequest/ResourceTagKeys]))

(s/def
  :cognitect.aws.budgets/TagKey
  (s/spec #(re-matches (re-pattern "[\\S\\s]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/PlannedBudgetLimits
  (s/map-of :cognitect.aws.budgets/GenericString :cognitect.aws.budgets/Spend))

(s/def :cognitect.aws.budgets/UpdateNotificationResponse (s/keys))

(s/def
  :cognitect.aws.budgets/UpdateSubscriberRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.UpdateSubscriberRequest/AccountId
     :cognitect.aws.budgets.UpdateSubscriberRequest/BudgetName
     :cognitect.aws.budgets.UpdateSubscriberRequest/Notification
     :cognitect.aws.budgets.UpdateSubscriberRequest/OldSubscriber
     :cognitect.aws.budgets.UpdateSubscriberRequest/NewSubscriber]))

(s/def
  :cognitect.aws.budgets/PolicyId
  (s/spec #(re-matches (re-pattern "^p-[0-9a-zA-Z_]{8,128}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/NotificationWithSubscribersList
  (s/coll-of :cognitect.aws.budgets/NotificationWithSubscribers :max-count 10))

(s/def
  :cognitect.aws.budgets/DimensionValue
  (s/spec #(re-matches (re-pattern "[\\S\\s]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.budgets/BudgetType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"SAVINGS_PLANS_UTILIZATION"
        "RI_UTILIZATION"
        "USAGE"
        "SAVINGS_PLANS_COVERAGE"
        "COST"
        "RI_COVERAGE"})))

(s/def :cognitect.aws.budgets/Notifications (s/coll-of :cognitect.aws.budgets/Notification))

(s/def
  :cognitect.aws.budgets/DescribeSubscribersForNotificationResponse
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.DescribeSubscribersForNotificationResponse/NextToken
     :cognitect.aws.budgets.DescribeSubscribersForNotificationResponse/Subscribers]))

(s/def
  :cognitect.aws.budgets/NotificationType
  (s/spec string? :gen #(s/gen #{"ACTUAL" "FORECASTED"})))

(s/def
  :cognitect.aws.budgets/DescribeNotificationsForBudgetResponse
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.DescribeNotificationsForBudgetResponse/NextToken
     :cognitect.aws.budgets.DescribeNotificationsForBudgetResponse/Notifications]))

(s/def
  :cognitect.aws.budgets/Role
  (s/spec
    #(re-matches (re-pattern "^([\\u0021-\\u007F]+\\u002F)?[\\w+=,.@-]+$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/NotificationWithSubscribers
  (s/keys
    :req-un
    [:cognitect.aws.budgets.NotificationWithSubscribers/Notification
     :cognitect.aws.budgets.NotificationWithSubscribers/Subscribers]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetsResponse/NextToken
     :cognitect.aws.budgets.DescribeBudgetsResponse/Budgets]))

(s/def :cognitect.aws.budgets/NotificationState (s/spec string? :gen #(s/gen #{"ALARM" "OK"})))

(s/def
  :cognitect.aws.budgets/ScpActionDefinition
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ScpActionDefinition/PolicyId
     :cognitect.aws.budgets.ScpActionDefinition/TargetIds]))

(s/def
  :cognitect.aws.budgets/BudgetName
  (s/spec
    #(re-matches (re-pattern "^(?![^:\\\\]*/action/|(?i).*<script>.*</script>.*)[^:\\\\]+$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.budgets/ThresholdType
  (s/spec string? :gen #(s/gen #{"ABSOLUTE_VALUE" "PERCENTAGE"})))

(s/def
  :cognitect.aws.budgets/Metric
  (s/spec
    string?
    :gen
    #(s/gen
      #{"AmortizedCost"
        "NetUnblendedCost"
        "NetAmortizedCost"
        "UsageQuantity"
        "BlendedCost"
        "UnblendedCost"
        "Hours"
        "NormalizedUsageAmount"})))

(s/def
  :cognitect.aws.budgets/Subscribers
  (s/coll-of :cognitect.aws.budgets/Subscriber :min-count 1 :max-count 11))

(s/def
  :cognitect.aws.budgets/ExpressionDimensionValues
  (s/keys
    :req-un
    [:cognitect.aws.budgets.ExpressionDimensionValues/Key
     :cognitect.aws.budgets.ExpressionDimensionValues/Values]
    :opt-un
    [:cognitect.aws.budgets.ExpressionDimensionValues/MatchOptions]))

(s/def
  :cognitect.aws.budgets/ListTagsForResourceResponse
  (s/keys :opt-un [:cognitect.aws.budgets.ListTagsForResourceResponse/ResourceTags]))

(s/def
  :cognitect.aws.budgets/CreateSubscriberRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.CreateSubscriberRequest/AccountId
     :cognitect.aws.budgets.CreateSubscriberRequest/BudgetName
     :cognitect.aws.budgets.CreateSubscriberRequest/Notification
     :cognitect.aws.budgets.CreateSubscriberRequest/Subscriber]))

(s/def
  :cognitect.aws.budgets/InvalidNextTokenException
  (s/keys :opt-un [:cognitect.aws.budgets.InvalidNextTokenException/Message]))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionsForAccountResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForAccountResponse/Actions]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionsForAccountResponse/NextToken]))

(s/def :cognitect.aws.budgets/Budgets (s/coll-of :cognitect.aws.budgets/Budget))

(s/def
  :cognitect.aws.budgets/DescribeBudgetActionHistoriesResponse
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DescribeBudgetActionHistoriesResponse/ActionHistories]
    :opt-un
    [:cognitect.aws.budgets.DescribeBudgetActionHistoriesResponse/NextToken]))

(s/def
  :cognitect.aws.budgets/DeleteNotificationRequest
  (s/keys
    :req-un
    [:cognitect.aws.budgets.DeleteNotificationRequest/AccountId
     :cognitect.aws.budgets.DeleteNotificationRequest/BudgetName
     :cognitect.aws.budgets.DeleteNotificationRequest/Notification]))

(s/def :cognitect.aws.budgets.ActionHistory/Timestamp :cognitect.aws.budgets/GenericTimestamp)

(s/def :cognitect.aws.budgets.ActionHistory/Status :cognitect.aws.budgets/ActionStatus)

(s/def :cognitect.aws.budgets.ActionHistory/EventType :cognitect.aws.budgets/EventType)

(s/def
  :cognitect.aws.budgets.ActionHistory/ActionHistoryDetails
  :cognitect.aws.budgets/ActionHistoryDetails)

(s/def :cognitect.aws.budgets.DuplicateRecordException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.ThrottlingException/Message :cognitect.aws.budgets/errorMessage)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/ApprovalModel
  :cognitect.aws.budgets/ApprovalModel)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/ActionThreshold
  :cognitect.aws.budgets/ActionThreshold)

(s/def :cognitect.aws.budgets.CreateBudgetActionRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/ActionType
  :cognitect.aws.budgets/ActionType)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/Definition
  :cognitect.aws.budgets/Definition)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/ExecutionRoleArn
  :cognitect.aws.budgets/RoleArn)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/NotificationType
  :cognitect.aws.budgets/NotificationType)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/Subscribers
  :cognitect.aws.budgets/Subscribers)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionRequest/ResourceTags
  :cognitect.aws.budgets/ResourceTagList)

(s/def
  :cognitect.aws.budgets.TagResourceRequest/ResourceARN
  :cognitect.aws.budgets/AmazonResourceName)

(s/def
  :cognitect.aws.budgets.TagResourceRequest/ResourceTags
  :cognitect.aws.budgets/ResourceTagList)

(s/def
  :cognitect.aws.budgets.ServiceQuotaExceededException/Message
  :cognitect.aws.budgets/errorMessage)

(s/def
  :cognitect.aws.budgets.ActionThreshold/ActionThresholdValue
  :cognitect.aws.budgets/NotificationThreshold)

(s/def
  :cognitect.aws.budgets.ActionThreshold/ActionThresholdType
  :cognitect.aws.budgets/ThresholdType)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForAccountRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.CreateBudgetActionResponse/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.CreateBudgetActionResponse/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.CreateBudgetActionResponse/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.ExecuteBudgetActionRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.ExecuteBudgetActionRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.ExecuteBudgetActionRequest/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.ExecuteBudgetActionRequest/ExecutionType
  :cognitect.aws.budgets/ExecutionType)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.DescribeBudgetActionRequest/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionResponse/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionResponse/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.DescribeBudgetActionResponse/Action :cognitect.aws.budgets/Action)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetResponse/Actions
  :cognitect.aws.budgets/Actions)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForBudgetResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.DeleteBudgetActionResponse/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DeleteBudgetActionResponse/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.DeleteBudgetActionResponse/Action :cognitect.aws.budgets/Action)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/Notification
  :cognitect.aws.budgets/Notification)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.AccessDeniedException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.Notification/NotificationType :cognitect.aws.budgets/NotificationType)

(s/def
  :cognitect.aws.budgets.Notification/ComparisonOperator
  :cognitect.aws.budgets/ComparisonOperator)

(s/def :cognitect.aws.budgets.Notification/Threshold :cognitect.aws.budgets/NotificationThreshold)

(s/def :cognitect.aws.budgets.Notification/ThresholdType :cognitect.aws.budgets/ThresholdType)

(s/def
  :cognitect.aws.budgets.Notification/NotificationState
  :cognitect.aws.budgets/NotificationState)

(s/def :cognitect.aws.budgets.CreateBudgetRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.CreateBudgetRequest/Budget :cognitect.aws.budgets/Budget)

(s/def
  :cognitect.aws.budgets.CreateBudgetRequest/NotificationsWithSubscribers
  :cognitect.aws.budgets/NotificationWithSubscribersList)

(s/def
  :cognitect.aws.budgets.CreateBudgetRequest/ResourceTags
  :cognitect.aws.budgets/ResourceTagList)

(s/def :cognitect.aws.budgets.CalculatedSpend/ActualSpend :cognitect.aws.budgets/Spend)

(s/def :cognitect.aws.budgets.CalculatedSpend/ForecastedSpend :cognitect.aws.budgets/Spend)

(s/def :cognitect.aws.budgets.ResourceTag/Key :cognitect.aws.budgets/ResourceTagKey)

(s/def :cognitect.aws.budgets.ResourceTag/Value :cognitect.aws.budgets/ResourceTagValue)

(s/def :cognitect.aws.budgets.TagValues/Key :cognitect.aws.budgets/TagKey)

(s/def :cognitect.aws.budgets.TagValues/Values :cognitect.aws.budgets/Values)

(s/def :cognitect.aws.budgets.TagValues/MatchOptions :cognitect.aws.budgets/MatchOptions)

(s/def :cognitect.aws.budgets.ActionHistoryDetails/Message :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.ActionHistoryDetails/Action :cognitect.aws.budgets/Action)

(s/def
  :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/MaxResults
  :cognitect.aws.budgets/MaxResultsBudgetNotifications)

(s/def
  :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.CostCategoryValues/Key :cognitect.aws.budgets/CostCategoryName)

(s/def :cognitect.aws.budgets.CostCategoryValues/Values :cognitect.aws.budgets/Values)

(s/def :cognitect.aws.budgets.CostCategoryValues/MatchOptions :cognitect.aws.budgets/MatchOptions)

(s/def :cognitect.aws.budgets.Budget/FilterExpression :cognitect.aws.budgets/Expression)

(s/def :cognitect.aws.budgets.Budget/CalculatedSpend :cognitect.aws.budgets/CalculatedSpend)

(s/def :cognitect.aws.budgets.Budget/CostFilters :cognitect.aws.budgets/CostFilters)

(s/def :cognitect.aws.budgets.Budget/LastUpdatedTime :cognitect.aws.budgets/GenericTimestamp)

(s/def :cognitect.aws.budgets.Budget/AutoAdjustData :cognitect.aws.budgets/AutoAdjustData)

(s/def :cognitect.aws.budgets.Budget/Metrics :cognitect.aws.budgets/Metrics)

(s/def :cognitect.aws.budgets.Budget/TimePeriod :cognitect.aws.budgets/TimePeriod)

(s/def :cognitect.aws.budgets.Budget/TimeUnit :cognitect.aws.budgets/TimeUnit)

(s/def :cognitect.aws.budgets.Budget/CostTypes :cognitect.aws.budgets/CostTypes)

(s/def :cognitect.aws.budgets.Budget/PlannedBudgetLimits :cognitect.aws.budgets/PlannedBudgetLimits)

(s/def :cognitect.aws.budgets.Budget/BudgetLimit :cognitect.aws.budgets/Spend)

(s/def :cognitect.aws.budgets.Budget/BudgetType :cognitect.aws.budgets/BudgetType)

(s/def :cognitect.aws.budgets.Budget/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.ExecuteBudgetActionResponse/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.ExecuteBudgetActionResponse/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.ExecuteBudgetActionResponse/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.ExecuteBudgetActionResponse/ExecutionType
  :cognitect.aws.budgets/ExecutionType)

(s/def
  :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountResponse/BudgetNotificationsForAccount
  :cognitect.aws.budgets/BudgetNotificationsForAccountList)

(s/def
  :cognitect.aws.budgets.DescribeBudgetNotificationsForAccountResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.ResourceLockedException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.AutoAdjustData/AutoAdjustType :cognitect.aws.budgets/AutoAdjustType)

(s/def
  :cognitect.aws.budgets.AutoAdjustData/HistoricalOptions
  :cognitect.aws.budgets/HistoricalOptions)

(s/def
  :cognitect.aws.budgets.AutoAdjustData/LastAutoAdjustTime
  :cognitect.aws.budgets/GenericTimestamp)

(s/def :cognitect.aws.budgets.NotFoundException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.UpdateNotificationRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.UpdateNotificationRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.UpdateNotificationRequest/OldNotification
  :cognitect.aws.budgets/Notification)

(s/def
  :cognitect.aws.budgets.UpdateNotificationRequest/NewNotification
  :cognitect.aws.budgets/Notification)

(s/def :cognitect.aws.budgets.DescribeBudgetsRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetsRequest/MaxResults
  :cognitect.aws.budgets/MaxResultsDescribeBudgets)

(s/def :cognitect.aws.budgets.DescribeBudgetsRequest/NextToken :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.DescribeBudgetsRequest/ShowFilterExpression
  :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.UpdateBudgetActionResponse/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionResponse/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.UpdateBudgetActionResponse/OldAction :cognitect.aws.budgets/Action)

(s/def :cognitect.aws.budgets.UpdateBudgetActionResponse/NewAction :cognitect.aws.budgets/Action)

(s/def :cognitect.aws.budgets.UpdateBudgetRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.UpdateBudgetRequest/NewBudget :cognitect.aws.budgets/Budget)

(s/def
  :cognitect.aws.budgets.HistoricalOptions/BudgetAdjustmentPeriod
  :cognitect.aws.budgets/AdjustmentPeriod)

(s/def
  :cognitect.aws.budgets.HistoricalOptions/LookBackAvailablePeriods
  :cognitect.aws.budgets/AdjustmentPeriod)

(s/def :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetType :cognitect.aws.budgets/BudgetType)

(s/def
  :cognitect.aws.budgets.BudgetPerformanceHistory/CostFilters
  :cognitect.aws.budgets/CostFilters)

(s/def :cognitect.aws.budgets.BudgetPerformanceHistory/CostTypes :cognitect.aws.budgets/CostTypes)

(s/def :cognitect.aws.budgets.BudgetPerformanceHistory/TimeUnit :cognitect.aws.budgets/TimeUnit)

(s/def
  :cognitect.aws.budgets.BudgetPerformanceHistory/BudgetedAndActualAmountsList
  :cognitect.aws.budgets/BudgetedAndActualAmountsList)

(s/def :cognitect.aws.budgets.DeleteSubscriberRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.DeleteSubscriberRequest/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DeleteSubscriberRequest/Notification
  :cognitect.aws.budgets/Notification)

(s/def :cognitect.aws.budgets.DeleteSubscriberRequest/Subscriber :cognitect.aws.budgets/Subscriber)

(s/def
  :cognitect.aws.budgets.Definition/IamActionDefinition
  :cognitect.aws.budgets/IamActionDefinition)

(s/def
  :cognitect.aws.budgets.Definition/ScpActionDefinition
  :cognitect.aws.budgets/ScpActionDefinition)

(s/def
  :cognitect.aws.budgets.Definition/SsmActionDefinition
  :cognitect.aws.budgets/SsmActionDefinition)

(s/def
  :cognitect.aws.budgets.BudgetNotificationsForAccount/Notifications
  :cognitect.aws.budgets/Notifications)

(s/def
  :cognitect.aws.budgets.BudgetNotificationsForAccount/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.Expression/Or :cognitect.aws.budgets/Expressions)

(s/def :cognitect.aws.budgets.Expression/And :cognitect.aws.budgets/Expressions)

(s/def :cognitect.aws.budgets.Expression/Not :cognitect.aws.budgets/Expression)

(s/def
  :cognitect.aws.budgets.Expression/Dimensions
  :cognitect.aws.budgets/ExpressionDimensionValues)

(s/def :cognitect.aws.budgets.Expression/Tags :cognitect.aws.budgets/TagValues)

(s/def :cognitect.aws.budgets.Expression/CostCategories :cognitect.aws.budgets/CostCategoryValues)

(s/def :cognitect.aws.budgets.ExpiredNextTokenException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.DeleteBudgetActionRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DeleteBudgetActionRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.DeleteBudgetActionRequest/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/ActionId
  :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/TimePeriod
  :cognitect.aws.budgets/TimePeriod)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.CreationLimitExceededException/Message
  :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.CreateNotificationRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.CreateNotificationRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.CreateNotificationRequest/Notification
  :cognitect.aws.budgets/Notification)

(s/def
  :cognitect.aws.budgets.CreateNotificationRequest/Subscribers
  :cognitect.aws.budgets/Subscribers)

(s/def
  :cognitect.aws.budgets.ListTagsForResourceRequest/ResourceARN
  :cognitect.aws.budgets/AmazonResourceName)

(s/def :cognitect.aws.budgets.DescribeBudgetRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.DescribeBudgetRequest/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeBudgetRequest/ShowFilterExpression
  :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.InvalidParameterException/Message :cognitect.aws.budgets/errorMessage)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/ApprovalModel
  :cognitect.aws.budgets/ApprovalModel)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/ActionThreshold
  :cognitect.aws.budgets/ActionThreshold)

(s/def :cognitect.aws.budgets.UpdateBudgetActionRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.UpdateBudgetActionRequest/ActionId :cognitect.aws.budgets/ActionId)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/Definition
  :cognitect.aws.budgets/Definition)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/ExecutionRoleArn
  :cognitect.aws.budgets/RoleArn)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/NotificationType
  :cognitect.aws.budgets/NotificationType)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.UpdateBudgetActionRequest/Subscribers
  :cognitect.aws.budgets/Subscribers)

(s/def :cognitect.aws.budgets.Spend/Amount :cognitect.aws.budgets/NumericValue)

(s/def :cognitect.aws.budgets.Spend/Unit :cognitect.aws.budgets/UnitValue)

(s/def :cognitect.aws.budgets.IamActionDefinition/PolicyArn :cognitect.aws.budgets/PolicyArn)

(s/def :cognitect.aws.budgets.IamActionDefinition/Roles :cognitect.aws.budgets/Roles)

(s/def :cognitect.aws.budgets.IamActionDefinition/Groups :cognitect.aws.budgets/Groups)

(s/def :cognitect.aws.budgets.IamActionDefinition/Users :cognitect.aws.budgets/Users)

(s/def
  :cognitect.aws.budgets.SsmActionDefinition/ActionSubType
  :cognitect.aws.budgets/ActionSubType)

(s/def :cognitect.aws.budgets.SsmActionDefinition/Region :cognitect.aws.budgets/Region)

(s/def :cognitect.aws.budgets.SsmActionDefinition/InstanceIds :cognitect.aws.budgets/InstanceIds)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/AccountId
  :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/TimePeriod
  :cognitect.aws.budgets/TimePeriod)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/MaxResults
  :cognitect.aws.budgets/MaxResults)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryRequest/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.DeleteBudgetRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.DeleteBudgetRequest/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.InternalErrorException/Message :cognitect.aws.budgets/errorMessage)

(s/def :cognitect.aws.budgets.TimePeriod/Start :cognitect.aws.budgets/GenericTimestamp)

(s/def :cognitect.aws.budgets.TimePeriod/End :cognitect.aws.budgets/GenericTimestamp)

(s/def :cognitect.aws.budgets.Action/ApprovalModel :cognitect.aws.budgets/ApprovalModel)

(s/def :cognitect.aws.budgets.Action/ActionThreshold :cognitect.aws.budgets/ActionThreshold)

(s/def :cognitect.aws.budgets.Action/ActionType :cognitect.aws.budgets/ActionType)

(s/def :cognitect.aws.budgets.Action/ActionId :cognitect.aws.budgets/ActionId)

(s/def :cognitect.aws.budgets.Action/Definition :cognitect.aws.budgets/Definition)

(s/def :cognitect.aws.budgets.Action/ExecutionRoleArn :cognitect.aws.budgets/RoleArn)

(s/def :cognitect.aws.budgets.Action/NotificationType :cognitect.aws.budgets/NotificationType)

(s/def :cognitect.aws.budgets.Action/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def :cognitect.aws.budgets.Action/Subscribers :cognitect.aws.budgets/Subscribers)

(s/def :cognitect.aws.budgets.Action/Status :cognitect.aws.budgets/ActionStatus)

(s/def :cognitect.aws.budgets.BudgetedAndActualAmounts/BudgetedAmount :cognitect.aws.budgets/Spend)

(s/def :cognitect.aws.budgets.BudgetedAndActualAmounts/ActualAmount :cognitect.aws.budgets/Spend)

(s/def :cognitect.aws.budgets.BudgetedAndActualAmounts/TimePeriod :cognitect.aws.budgets/TimePeriod)

(s/def :cognitect.aws.budgets.CostTypes/IncludeRefund :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeSubscription :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeUpfront :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeRecurring :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeDiscount :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeSupport :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeCredit :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/UseAmortized :cognitect.aws.budgets/NullableBoolean)

(s/def
  :cognitect.aws.budgets.CostTypes/IncludeOtherSubscription
  :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/UseBlended :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.CostTypes/IncludeTax :cognitect.aws.budgets/NullableBoolean)

(s/def :cognitect.aws.budgets.Subscriber/SubscriptionType :cognitect.aws.budgets/SubscriptionType)

(s/def :cognitect.aws.budgets.Subscriber/Address :cognitect.aws.budgets/SubscriberAddress)

(s/def :cognitect.aws.budgets.DescribeBudgetResponse/Budget :cognitect.aws.budgets/Budget)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryResponse/BudgetPerformanceHistory
  :cognitect.aws.budgets/BudgetPerformanceHistory)

(s/def
  :cognitect.aws.budgets.DescribeBudgetPerformanceHistoryResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.UntagResourceRequest/ResourceARN
  :cognitect.aws.budgets/AmazonResourceName)

(s/def
  :cognitect.aws.budgets.UntagResourceRequest/ResourceTagKeys
  :cognitect.aws.budgets/ResourceTagKeyList)

(s/def :cognitect.aws.budgets.UpdateSubscriberRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.UpdateSubscriberRequest/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.UpdateSubscriberRequest/Notification
  :cognitect.aws.budgets/Notification)

(s/def
  :cognitect.aws.budgets.UpdateSubscriberRequest/OldSubscriber
  :cognitect.aws.budgets/Subscriber)

(s/def
  :cognitect.aws.budgets.UpdateSubscriberRequest/NewSubscriber
  :cognitect.aws.budgets/Subscriber)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationResponse/Subscribers
  :cognitect.aws.budgets/Subscribers)

(s/def
  :cognitect.aws.budgets.DescribeSubscribersForNotificationResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetResponse/Notifications
  :cognitect.aws.budgets/Notifications)

(s/def
  :cognitect.aws.budgets.DescribeNotificationsForBudgetResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.NotificationWithSubscribers/Notification
  :cognitect.aws.budgets/Notification)

(s/def
  :cognitect.aws.budgets.NotificationWithSubscribers/Subscribers
  :cognitect.aws.budgets/Subscribers)

(s/def :cognitect.aws.budgets.DescribeBudgetsResponse/Budgets :cognitect.aws.budgets/Budgets)

(s/def
  :cognitect.aws.budgets.DescribeBudgetsResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.ScpActionDefinition/PolicyId :cognitect.aws.budgets/PolicyId)

(s/def :cognitect.aws.budgets.ScpActionDefinition/TargetIds :cognitect.aws.budgets/TargetIds)

(s/def :cognitect.aws.budgets.ExpressionDimensionValues/Key :cognitect.aws.budgets/Dimension)

(s/def :cognitect.aws.budgets.ExpressionDimensionValues/Values :cognitect.aws.budgets/Values)

(s/def
  :cognitect.aws.budgets.ExpressionDimensionValues/MatchOptions
  :cognitect.aws.budgets/MatchOptions)

(s/def
  :cognitect.aws.budgets.ListTagsForResourceResponse/ResourceTags
  :cognitect.aws.budgets/ResourceTagList)

(s/def :cognitect.aws.budgets.CreateSubscriberRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def :cognitect.aws.budgets.CreateSubscriberRequest/BudgetName :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.CreateSubscriberRequest/Notification
  :cognitect.aws.budgets/Notification)

(s/def :cognitect.aws.budgets.CreateSubscriberRequest/Subscriber :cognitect.aws.budgets/Subscriber)

(s/def :cognitect.aws.budgets.InvalidNextTokenException/Message :cognitect.aws.budgets/errorMessage)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForAccountResponse/Actions
  :cognitect.aws.budgets/Actions)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionsForAccountResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesResponse/ActionHistories
  :cognitect.aws.budgets/ActionHistories)

(s/def
  :cognitect.aws.budgets.DescribeBudgetActionHistoriesResponse/NextToken
  :cognitect.aws.budgets/GenericString)

(s/def :cognitect.aws.budgets.DeleteNotificationRequest/AccountId :cognitect.aws.budgets/AccountId)

(s/def
  :cognitect.aws.budgets.DeleteNotificationRequest/BudgetName
  :cognitect.aws.budgets/BudgetName)

(s/def
  :cognitect.aws.budgets.DeleteNotificationRequest/Notification
  :cognitect.aws.budgets/Notification)

